<?php
/*
 * 파일명: class-pixabay-api.php
 * 위치: /wp-content/plugins/image-format-converter/includes/
 * 기능: Pixabay API 통합 및 이미지 검색/다운로드 처리
 */

// 직접 접근 방지
if (!defined('ABSPATH')) {
    exit;
}

class Pixabay_API {
    /**
	 * 클래스 생성자
	 */
	public function __construct() {
		// AJAX 처리 액션 등록
		add_action('wp_ajax_img_manage_pixabay_search', array($this, 'ajax_pixabay_search'));
		add_action('wp_ajax_img_manage_pixabay_download', array($this, 'ajax_pixabay_download'));
		
		// Gutenberg 블록 에디터를 위한 REST API 등록
		add_action('rest_api_init', array($this, 'register_rest_routes'));
		
		// Pixabay 이미지 모달 템플릿 추가
		add_action('admin_footer', array($this, 'pixabay_modal_template'));
		
		// 클래식 에디터용 버튼 추가
		add_action('media_buttons', array($this, 'add_classic_editor_button'), 15);
		
		// 툴바 버튼 추가를 위한 관리자 스크립트 추가
		add_action('admin_enqueue_scripts', array($this, 'add_pixabay_button_script'));
	}
	
	
	/**
	 * 클래식 에디터용 버튼 추가
	 */
	public function add_classic_editor_button() {
		// 설정 확인
		$options = get_option('img_manage_options', array());
		if (!isset($options['pixabay_enabled']) || $options['pixabay_enabled'] != 1) {
			return;
		}
		
		// 버튼 출력
		echo '<button type="button" id="pixabay-button-classic" class="button" style="padding-left: 0.4em;">
			<span class="wp-media-buttons-icon dashicons dashicons-format-image" style="margin-top: 2px;"></span>
			Pixabay 이미지 검색
		</button>';
		
		// 버튼 클릭 이벤트
		?>
		<script>
		jQuery(document).ready(function($) {
			$('#pixabay-button-classic').on('click', function(e) {
				e.preventDefault();
				const event = new CustomEvent('open-pixabay-modal');
				document.dispatchEvent(event);
			});
		});
		</script>
		<?php
	}
		
		/**
	 * 추가 스크립트 로드 (Gutenberg용 버튼 추가)
	 */
	public function add_pixabay_button_script($hook) {
		if ($hook !== 'post.php' && $hook !== 'post-new.php') {
			return;
		}
		
		// 설정 확인
		$options = get_option('img_manage_options', array());
		if (!isset($options['pixabay_enabled']) || $options['pixabay_enabled'] != 1) {
			return;
		}
		
		// 인라인 스크립트 추가
		wp_add_inline_script('wp-edit-post', '
			jQuery(function($) {
				// 툴바에 픽사베이 버튼 추가
				var addPixabayButton = function() {
					if ($(".pixabay-toolbar-button").length) return;
					
					var toolbar = $(".edit-post-header-toolbar");
					if (!toolbar.length) return;
					
					var button = $("<button/>", {
						class: "components-button is-primary pixabay-toolbar-button",
						text: "Pixabay 이미지",
						css: { marginLeft: "8px" },
						click: function() {
							var event = new CustomEvent("open-pixabay-modal");
							document.dispatchEvent(event);
						}
					});
					
					toolbar.append(button);
				};
				
				// 편집기 로드 후 버튼 추가
				setTimeout(addPixabayButton, 1000);
				setInterval(addPixabayButton, 1000);
			});
		', 'after');
	}

    
    /**
     * REST API 라우트 등록
     */
    public function register_rest_routes() {
        register_rest_route('img-manage/v1', '/pixabay-search', array(
            'methods' => 'GET',
            'callback' => array($this, 'rest_pixabay_search'),
            'permission_callback' => function() {
                return current_user_can('edit_posts');
            }
        ));
        
        register_rest_route('img-manage/v1', '/pixabay-download', array(
            'methods' => 'POST',
            'callback' => array($this, 'rest_pixabay_download'),
            'permission_callback' => function() {
                return current_user_can('upload_files');
            }
        ));
    }
    
    /**
     * 클래식 에디터에 버튼 추가
     */
    public function add_media_button() {
        // 설정 확인
        $options = get_option('img_manage_options', array());
        if (!isset($options['pixabay_enabled']) || $options['pixabay_enabled'] != 1) {
            return;
        }
        
        // 버튼 출력
        echo '<button type="button" id="pixabay-search-button-classic" class="button" style="padding-left: 0.4em;">
            <span class="wp-media-buttons-icon dashicons dashicons-format-image" style="margin-top: 2px;"></span>
            Pixabay 이미지 검색
        </button>';
        
        // 버튼 클릭 시 이벤트
        ?>
        <script>
        jQuery(document).ready(function($) {
            $('#pixabay-search-button-classic').on('click', function(e) {
                e.preventDefault();
                // 모달 열기 이벤트 발생
                const event = new CustomEvent('open-pixabay-modal');
                document.dispatchEvent(event);
            });
        });
        </script>
        <?php
    }
    
    /**
     * Pixabay 검색 REST API 콜백
     */
    public function rest_pixabay_search($request) {
        $params = $request->get_params();
        $query = sanitize_text_field($params['query'] ?? '');
        $image_type = sanitize_text_field($params['image_type'] ?? 'photo');
        $orientation = sanitize_text_field($params['orientation'] ?? 'all');
        $category = sanitize_text_field($params['category'] ?? '');
        $per_page = intval($params['per_page'] ?? 20);
        $page = intval($params['page'] ?? 1);
        
        $results = $this->search_pixabay_images($query, $image_type, $orientation, $category, $per_page, $page);
        
        if (is_wp_error($results)) {
            return new WP_REST_Response(array(
                'success' => false,
                'message' => $results->get_error_message()
            ), 500);
        }
        
        return new WP_REST_Response(array(
            'success' => true,
            'data' => $results
        ), 200);
    }
    
    /**
     * Pixabay 이미지 다운로드 REST API 콜백
     */
    public function rest_pixabay_download($request) {
        $params = $request->get_params();
        $image_url = sanitize_url($params['image_url'] ?? '');
        $pixabay_id = intval($params['pixabay_id'] ?? 0);
        $alt_text = sanitize_text_field($params['alt_text'] ?? '');
        $filename = sanitize_file_name($params['filename'] ?? '');
        
        if (empty($image_url) || empty($pixabay_id)) {
            return new WP_REST_Response(array(
                'success' => false,
                'message' => '이미지 URL 또는 Pixabay ID가 누락되었습니다.'
            ), 400);
        }
        
        $attachment_id = $this->download_pixabay_image($image_url, $pixabay_id, $alt_text, $filename);
        
        if (is_wp_error($attachment_id)) {
            return new WP_REST_Response(array(
                'success' => false,
                'message' => $attachment_id->get_error_message()
            ), 500);
        }
        
        $attachment_url = wp_get_attachment_url($attachment_id);
        $attachment_data = wp_prepare_attachment_for_js($attachment_id);
        
        return new WP_REST_Response(array(
            'success' => true,
            'attachment_id' => $attachment_id,
            'url' => $attachment_url,
            'data' => $attachment_data
        ), 200);
    }
    
    /**
     * AJAX 검색 처리 콜백
     */
    public function ajax_pixabay_search() {
        // 보안 확인
        if (!check_ajax_referer('img-manage-pixabay', 'nonce', false)) {
            wp_send_json_error('보안 검증에 실패했습니다.');
        }
        
        if (!current_user_can('edit_posts')) {
            wp_send_json_error('충분한 권한이 없습니다.');
        }
        
        $query = sanitize_text_field($_GET['query'] ?? '');
        $image_type = sanitize_text_field($_GET['image_type'] ?? 'photo');
        $orientation = sanitize_text_field($_GET['orientation'] ?? 'all');
        $category = sanitize_text_field($_GET['category'] ?? '');
        $per_page = intval($_GET['per_page'] ?? 20);
        $page = intval($_GET['page'] ?? 1);
        
        $results = $this->search_pixabay_images($query, $image_type, $orientation, $category, $per_page, $page);
        
        if (is_wp_error($results)) {
            wp_send_json_error($results->get_error_message());
        }
        
        wp_send_json_success($results);
    }
    
    /**
     * AJAX 다운로드 처리 콜백
     */
    public function ajax_pixabay_download() {
        // 보안 확인
        if (!check_ajax_referer('img-manage-pixabay', 'nonce', false)) {
            wp_send_json_error('보안 검증에 실패했습니다.');
        }
        
        if (!current_user_can('upload_files')) {
            wp_send_json_error('충분한 권한이 없습니다.');
        }
        
        $image_url = sanitize_url($_POST['image_url'] ?? '');
        $pixabay_id = intval($_POST['pixabay_id'] ?? 0);
        $alt_text = sanitize_text_field($_POST['alt_text'] ?? '');
        $filename = sanitize_file_name($_POST['filename'] ?? '');
        
        if (empty($image_url) || empty($pixabay_id)) {
            wp_send_json_error('이미지 URL 또는 Pixabay ID가 누락되었습니다.');
        }
        
        $attachment_id = $this->download_pixabay_image($image_url, $pixabay_id, $alt_text, $filename);
        
        if (is_wp_error($attachment_id)) {
            wp_send_json_error($attachment_id->get_error_message());
        }
        
        $attachment_url = wp_get_attachment_url($attachment_id);
        $attachment_data = wp_prepare_attachment_for_js($attachment_id);
        
        wp_send_json_success(array(
            'attachment_id' => $attachment_id,
            'url' => $attachment_url,
            'data' => $attachment_data
        ));
    }
    
    /**
	 * Pixabay API를 통해 이미지 검색
	 */
	public function search_pixabay_images($query, $image_type = 'photo', $orientation = 'all', $category = '', $per_page = 20, $page = 1) {
		$api_url = 'https://pixabay.com/api/';
		
		// 옵션에서 API 키 가져오기
		$options = get_option('img_manage_options');
		$api_key = isset($options['pixabay_api_key']) ? $options['pixabay_api_key'] : '';
		
		// API 키가 없는 경우 오류 반환
		if (empty($api_key)) {
			return new WP_Error('pixabay_api_error', 'Pixabay API 키가 설정되지 않았습니다.');
		}
		
		// API 요청 파라미터 구성
		$params = array(
			'key' => $api_key,
			'q' => urlencode($query),
			'image_type' => $image_type,
			'per_page' => $per_page,
			'page' => $page,
			'safesearch' => true,
			'lang' => 'ko' // 한국어 검색 결과 (필요에 따라 변경 가능)
		);
		
		// 추가 파라미터 (값이 있을 경우에만 추가)
		if (!empty($orientation) && $orientation !== 'all') {
			$params['orientation'] = $orientation;
		}
		
		if (!empty($category)) {
			$params['category'] = $category;
		}
		
		// API 요청 URL 생성
		$request_url = add_query_arg($params, $api_url);
		
		// API 요청
		$response = wp_remote_get($request_url);
		
		// 오류 확인
		if (is_wp_error($response)) {
			return $response;
		}
		
		$response_code = wp_remote_retrieve_response_code($response);
		if ($response_code !== 200) {
			return new WP_Error('pixabay_api_error', '픽사베이 API 오류: ' . $response_code);
		}
		
		// 응답 바디 처리
		$body = wp_remote_retrieve_body($response);
		$data = json_decode($body, true);
		
		if (!$data || !isset($data['hits'])) {
			return new WP_Error('pixabay_api_error', '픽사베이 API 응답 데이터 오류');
		}
		
		return $data;
	}
    
    /**
     * Pixabay 이미지 다운로드 및 미디어 라이브러리에 추가
     */
    public function download_pixabay_image($image_url, $pixabay_id, $alt_text = '', $filename = '') {
        // 파일 확장자 추출
        $file_info = wp_check_filetype(basename($image_url));
        $ext = empty($file_info['ext']) ? 'jpg' : $file_info['ext'];
        
        // 파일명 생성
        if (empty($filename)) {
            $filename = 'pixabay-' . $pixabay_id;
        }
        
        // 임시 다운로드
        $temp_file = download_url($image_url);
        
        if (is_wp_error($temp_file)) {
            return $temp_file;
        }
        
        // 파일 정보 설정
        $file_array = array(
            'name' => $filename . '.' . $ext,
            'tmp_name' => $temp_file,
            'type' => $file_info['type'],
            'error' => 0,
            'size' => filesize($temp_file),
        );
        
        // 미디어 라이브러리에 추가
        $attachment_id = media_handle_sideload($file_array, 0, $alt_text);
        
        // 임시 파일 삭제
        @unlink($temp_file);
        
        if (is_wp_error($attachment_id)) {
            return $attachment_id;
        }
        
        // 이미지 메타데이터 업데이트
        $this->update_pixabay_image_meta($attachment_id, $pixabay_id, $alt_text);
        
        return $attachment_id;
    }
    
    /**
     * Pixabay 이미지 메타데이터 업데이트
     */
    private function update_pixabay_image_meta($attachment_id, $pixabay_id, $alt_text = '') {
        // alt 텍스트 설정
        if (!empty($alt_text)) {
            update_post_meta($attachment_id, '_wp_attachment_image_alt', $alt_text);
        }
        
        // Pixabay ID 저장 (추적을 위한 용도)
        update_post_meta($attachment_id, '_pixabay_image_id', $pixabay_id);
        
        // 소스 URL 저장 (저작권 정보)
        update_post_meta($attachment_id, '_pixabay_source_url', 'https://pixabay.com/get/' . $pixabay_id);
        
        // SEO 친화적인 이미지 설명 업데이트
        $attachment = array(
            'ID' => $attachment_id,
            'post_excerpt' => '이 이미지는 Pixabay에서 제공되는 무료 이미지입니다. ID: ' . $pixabay_id,
        );
        
        wp_update_post($attachment);
    }
    
    /**
     * Pixabay 이미지 검색 모달 템플릿 추가
     */
    public function pixabay_modal_template() {
        // 글 작성/편집 화면인지 확인
        $screen = get_current_screen();
        if (!$screen || !in_array($screen->base, array('post', 'post-new'))) {
            return;
        }
        
        // 설정 가져오기
        $options = get_option('img_manage_options');
        if (!isset($options['pixabay_enabled']) || $options['pixabay_enabled'] != 1) {
            return;
        }
        
        // 픽사베이 모달 HTML
        ?>
        <div id="pixabay-modal" class="pixabay-modal">
            <div class="pixabay-modal-content">
                <div class="pixabay-modal-header">
                    <span class="pixabay-modal-close">&times;</span>
                    <h2>Pixabay 무료 이미지 검색</h2>
                </div>
                
                <div class="pixabay-modal-body">
                    <div class="pixabay-search-container">
                        <div class="pixabay-search-form">
                            <div class="pixabay-search-input-group">
                                <input type="text" id="pixabay-search-input" placeholder="검색어를 입력하세요...">
                                <button id="pixabay-search-button" class="button button-primary">검색</button>
                            </div>
                            
                            <div class="pixabay-search-filters">
                                <div class="pixabay-filter-group">
                                    <label for="pixabay-image-type">이미지 유형:</label>
                                    <select id="pixabay-image-type">
                                        <option value="photo">사진</option>
                                        <option value="illustration">일러스트레이션</option>
                                        <option value="vector">벡터 그래픽</option>
                                        <option value="all">모든 이미지</option>
                                    </select>
                                </div>
                                
                                <div class="pixabay-filter-group">
                                    <label for="pixabay-orientation">방향:</label>
                                    <select id="pixabay-orientation">
                                        <option value="all">모든 방향</option>
                                        <option value="horizontal">가로 방향</option>
                                        <option value="vertical">세로 방향</option>
                                    </select>
                                </div>
                                
                                <div class="pixabay-filter-group">
                                    <label for="pixabay-category">카테고리:</label>
                                    <select id="pixabay-category">
                                        <option value="">카테고리 선택 없음</option>
                                        <option value="backgrounds">배경</option>
                                        <option value="fashion">패션</option>
                                        <option value="nature">자연</option>
                                        <option value="science">과학</option>
                                        <option value="education">교육</option>
                                        <option value="health">건강</option>
                                        <option value="people">사람</option>
                                        <option value="religion">종교</option>
                                        <option value="places">장소</option>
                                        <option value="animals">동물</option>
                                        <option value="industry">산업</option>
                                        <option value="computer">컴퓨터</option>
                                        <option value="food">음식</option>
                                        <option value="sports">스포츠</option>
                                        <option value="transportation">교통</option>
                                        <option value="travel">여행</option>
                                        <option value="buildings">건물</option>
                                        <option value="business">비즈니스</option>
                                        <option value="music">음악</option>
                                    </select>
                                </div>
                            </div>
                        </div>
                        
                        <div class="pixabay-search-results">
                            <div class="pixabay-results-info">
                                <span id="pixabay-results-count"></span>
                            </div>
                            
                            <div id="pixabay-images-container" class="pixabay-images-grid"></div>
                            
                            <div class="pixabay-load-more">
                                <button id="pixabay-load-more-button" class="button button-secondary">더 많은 이미지 불러오기</button>
                            </div>
                        </div>
                        
                        <div id="pixabay-loading" class="pixabay-loading">
                            <span class="spinner is-active"></span>
                            <span class="loading-text">이미지 로딩 중...</span>
                        </div>
                        
                        <div id="pixabay-no-results" class="pixabay-no-results">
                            검색 결과가 없습니다. 다른 검색어로 시도해보세요.
                        </div>
                    </div>
                    
                    <div class="pixabay-attribution">
                        이미지 제공: <a href="https://pixabay.com/" target="_blank" rel="noopener noreferrer">Pixabay</a>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }
}