<?php
/*
 * 파일명: class-image-converter.php
 * 위치: /wp-content/plugins/image-format-converter/includes/
 * 기능: 이미지 변환 및 최적화 핵심 기능
 * 작성일: 2025-04-21
 */

// 직접 접근 방지
if (!defined('ABSPATH')) {
    exit;
}

class Image_Converter {
    /**
     * 클래스 생성자
     */
    public function __construct() {
        // 기본 이미지 업로드 필터 추가
        add_filter('wp_handle_upload_prefilter', array($this, 'handle_upload_filter'));
        
        // 작은 이미지 처리를 위한 추가 필터
        add_filter('wp_handle_upload', array($this, 'small_image_filter'), 9);
        
        // 이미지 변환 메인 함수 필터
        add_filter('wp_handle_upload', array($this, 'convert_image_to_target'));
        
        // 업로드 전 이미지 처리를 위한 추가 필터
        add_filter('wp_handle_upload_prefilter', array($this, 'pre_upload'));
        
        // 미디어 업로드 완료 후 처리
        add_filter('wp_generate_attachment_metadata', array($this, 'after_upload'), 10, 2);
        
        // srcset 속성 추가 함수
        add_filter('post_thumbnail_html', array($this, 'add_srcset'), 10, 5);
        
        // 이미지 alt 태그 개선 함수
        add_action('save_post', array($this, 'improve_alt_tags'));
        add_action('add_attachment', array($this, 'set_alt_on_upload'));
        add_filter('the_content', array($this, 'title_to_alt'));
        add_filter('post_thumbnail_html', array($this, 'filter_thumbnail_alt'), 20, 5);
        add_filter('the_content', array($this, 'custom_thumbnail_alt'), 10);
    }
    
    /**
     * 기본 이미지 업로드 필터
     */
    public function handle_upload_filter($file) {
        // 이미지 파일이 아닌 경우 처리하지 않음
        if (strpos($file['type'], 'image/') !== 0) {
            return $file;
        }
        
        // 파일 경로와 URL 정보 로깅
        error_log('기본 업로드 필터: ' . $file['name'] . ' (타입: ' . $file['type'] . ')');
        
        return $file;
    }
    
    /**
     * 작은 이미지 처리를 위한 추가 필터
     */
    public function small_image_filter($data) {
        // 이미지 파일이 아닌 경우 처리하지 않음
        if (!isset($data['type']) || strpos($data['type'], 'image/') !== 0) {
            return $data;
        }
        
        error_log('작은 이미지 필터: ' . $data['file'] . ' (타입: ' . $data['type'] . ')');
        
        // 파일 정보 가져오기
        $file_path = $data['file'];
        
        // 파일이 존재하는지 확인
        if (!file_exists($file_path)) {
            error_log('파일이 존재하지 않음: ' . $file_path);
            return $data;
        }
        
        // 이미지 크기 확인
        $image_size = getimagesize($file_path);
        if (!$image_size) {
            error_log('이미지 크기를 가져올 수 없음: ' . $file_path);
            return $data;
        }
        
        $width = $image_size[0];
        $height = $image_size[1];
        $mime = $image_size['mime'];
        
        error_log('이미지 크기: ' . $width . 'x' . $height . ' (MIME: ' . $mime . ')');
        
        return $data;
    }
    
    /**
     * 이미지 변환 메인 함수
     */
    public function convert_image_to_target($upload) {
        // 디버깅 로그 추가
        error_log('이미지 변환 시작: ' . $upload['file'] . ' (타입: ' . $upload['type'] . ')');
        
        // 관리자 설정 옵션 가져오기
        $options = get_option('img_manage_options');
        
        $file_path = $upload['file'];
        $file_info = pathinfo($file_path);
        $ext = strtolower($file_info['extension']);
        
        // 처리 대상 확장자: png, jpg, jpeg, webp
        if (!in_array($ext, array('png', 'jpg', 'jpeg', 'webp'))) {
            error_log('지원하지 않는 확장자: ' . $ext);
            return $upload;
        }
        
        // 파일 존재 확인
        if (!file_exists($file_path)) {
            error_log('파일을 찾을 수 없음: ' . $file_path);
            return $upload;
        }
        
        // 파일 크기 로깅
        $filesize = filesize($file_path);
        error_log('파일 크기: ' . $filesize . ' 바이트');
        
        // 이미지 크기 확인
        $image_size = getimagesize($file_path);
        if (!$image_size) {
            error_log('이미지 크기를 가져올 수 없음: ' . $file_path);
            return $upload;
        }
        
        $width = $image_size[0];
        $height = $image_size[1];
        $mime = $image_size['mime'];
        
        error_log('이미지 크기: ' . $width . 'x' . $height . ' (MIME: ' . $mime . ')');
        
        try {
            // 초기화
            $image = false;
            $target = '';
            $quality = 0;
            
            // 이미지 타입별 설정 가져오기
            if ($ext === 'png') {
                if (!isset($options['png_enabled']) || intval($options['png_enabled']) !== 1) {
                    error_log('PNG 변환이 비활성화되어 있습니다.');
                    return $upload;
                }
                
                // PNG 이미지 생성 - 팔레트 모드 처리 추가
                $image = @imagecreatefrompng($file_path);
                
                // 팔레트 모드 PNG를 트루컬러로 변환
                if ($image) {
                    $truecolor = imagecreatetruecolor($width, $height);
                    imagealphablending($truecolor, false);
                    imagesavealpha($truecolor, true);
                    imagecopy($truecolor, $image, 0, 0, 0, 0, $width, $height);
                    imagedestroy($image);
                    $image = $truecolor;
                }
                
                $target = isset($options['png_target']) ? $options['png_target'] : 'webp';
                $quality = isset($options['png_quality']) ? intval($options['png_quality']) : 80;
            } elseif ($ext === 'jpg' || $ext === 'jpeg') {
                if (!isset($options['jpg_enabled']) || intval($options['jpg_enabled']) !== 1) {
                    error_log('JPG 변환이 비활성화되어 있습니다.');
                    return $upload;
                }
                $image = @imagecreatefromjpeg($file_path);
                $target = 'webp';
                $quality = isset($options['jpg_quality']) ? intval($options['jpg_quality']) : 85;
            } elseif ($ext === 'webp') {
                if (!isset($options['webp_enabled']) || intval($options['webp_enabled']) !== 1) {
                    error_log('WEBP 변환이 비활성화되어 있습니다.');
                    return $upload;
                }
                if (function_exists('imagecreatefromwebp')) {
                    $image = @imagecreatefromwebp($file_path);
                } else {
                    error_log('WebP 지원 함수 없음');
                    return $upload;
                }
                $target = 'jpeg';
                $quality = isset($options['webp_quality']) ? intval($options['webp_quality']) : 85;
            }
            
            if (!$image) {
                error_log('이미지 생성 실패: ' . $file_path);
                return $upload;
            }
            
            // 기본 설정 가져오기
            $max_width = isset($options['max_image_width']) ? intval($options['max_image_width']) : 1920;
            
            error_log('이미지 리사이징 필요 없음: ' . $width . 'px (최대: ' . $max_width . 'px)');
            
            // SEO 친화적인 파일명 생성
            $seo_filename = (isset($options['seo_filename']) && intval($options['seo_filename']) === 1);
            if ($seo_filename) {
                // 원본 파일명 유지하면서 고유성 보장
                $unique_filename = sanitize_file_name($file_info['filename']) . '-' . substr(md5(uniqid()), 0, 6);
            } else {
                // 기존 방식 (타임스탬프 + 랜덤 문자열)
                $unique_filename = $file_info['filename'] . '-' . time() . '-' . substr(md5(rand()), 0, 6);
            }
            
            // 리사이즈: 이미지 가로가 max_width보다 큰 경우에만
            if ($max_width > 0 && $width > $max_width) {
                error_log('이미지 리사이징: ' . $width . 'px -> ' . $max_width . 'px');
                $new_width = $max_width;
                $new_height = intval($height * ($max_width / $width));
                $resized = imagecreatetruecolor($new_width, $new_height);
                
                // PNG 투명도 보존
                if ($ext === 'png') {
                    imagecolortransparent($resized, imagecolorallocate($resized, 0, 0, 0));
                    imagealphablending($resized, false);
                    imagesavealpha($resized, true);
                } else {
                    // 흰색 배경 설정 (JPEG, WEBP)
                    $white = imagecolorallocate($resized, 255, 255, 255);
                    imagefill($resized, 0, 0, $white);
                }
                
                // 이미지 리사이즈
                imagecopyresampled($resized, $image, 0, 0, 0, 0, $new_width, $new_height, $width, $height);
                imagedestroy($image);
                $image = $resized;
                $width = $new_width;
                $height = $new_height;
            }
            
            $new_file_path = '';
            $new_mime = '';
            
            if ($target === 'webp') {
                if (function_exists('imagewebp')) {
                    $new_file_path = $file_info['dirname'] . '/' . $unique_filename . '.webp';
                    error_log('WebP로 변환 시도: ' . $new_file_path);
                    
                    // WebP 변환 시도
                    $success = imagewebp($image, $new_file_path, $quality);
                    
                    if (!$success || filesize($new_file_path) === 0) {
                        error_log('WebP 변환 실패, JPEG로 폴백');
                        // WebP 변환 실패 시 JPEG로 폴백
                        $new_file_path = $file_info['dirname'] . '/' . $unique_filename . '.jpg';
                        imagejpeg($image, $new_file_path, $quality);
                        $new_mime = 'image/jpeg';
                    } else {
                        $new_mime = 'image/webp';
                    }
                } else {
                    // WebP 지원이 없으면 fallback: JPEG
                    $new_file_path = $file_info['dirname'] . '/' . $unique_filename . '.jpg';
                    error_log('WebP 지원 없음, JPEG로 변환: ' . $new_file_path);
                    imagejpeg($image, $new_file_path, $quality);
                    $new_mime = 'image/jpeg';
                }
            } elseif ($target === 'jpeg' || $target === 'jpg') {
                $new_file_path = $file_info['dirname'] . '/' . $unique_filename . '.jpg';
                error_log('JPEG로 변환: ' . $new_file_path);
                imagejpeg($image, $new_file_path, $quality);
                $new_mime = 'image/jpeg';
            } else {
                // 지원하지 않는 타겟 형식일 경우
                imagedestroy($image);
                error_log('지원하지 않는 타겟 형식: ' . $target);
                return $upload;
            }
            
            // 메모리 해제
            imagedestroy($image);
            
            // 변환된 파일 확인
            if (!file_exists($new_file_path) || filesize($new_file_path) === 0) {
                error_log('변환된 파일이 생성되지 않거나 크기가 0입니다: ' . $new_file_path);
                return $upload;
            }
            
            $new_filesize = filesize($new_file_path);
            error_log('변환된 파일 크기: ' . $new_filesize . ' 바이트');
            
            // 변환된 이미지의 크기 정보 가져오기
            $new_image_size = getimagesize($new_file_path);
            if (!$new_image_size) {
                error_log('변환된 이미지 크기를 가져올 수 없음: ' . $new_file_path);
            } else {
                $new_width = $new_image_size[0];
                $new_height = $new_image_size[1];
                error_log('변환된 이미지 크기: ' . $new_width . 'x' . $new_height);
            }
            
            // 원본 파일 삭제 (변환 성공 후)
            if (file_exists($file_path)) {
                unlink($file_path);
                error_log('원본 파일 삭제: ' . $file_path);
            }
            
            // 업로드 정보 업데이트
            $upload['file'] = $new_file_path;
            $upload['url'] = str_replace($file_info['basename'], pathinfo($new_file_path, PATHINFO_BASENAME), $upload['url']);
            $upload['type'] = $new_mime;
            
            // 중요: 이미지 크기 정보 추가 (워드프레스 메타데이터 생성에 필요)
            if ($new_image_size) {
                $upload['width'] = $new_width;
                $upload['height'] = $new_height;
            }
            
            error_log('이미지 변환 성공: ' . $new_file_path);
            return $upload;
            
        } catch (Exception $e) {
            error_log('이미지 변환 중 오류 발생: ' . $e->getMessage());
            return $upload;
        }
    }
    
    /**
     * 업로드 전 이미지 처리
     */
    public function pre_upload($file) {
        error_log('업로드 전 이미지 처리: ' . $file['name'] . ' (타입: ' . $file['type'] . ')');
        return $file;
    }
    
    /**
     * 미디어 업로드 완료 후 처리
     */
    public function after_upload($metadata, $attachment_id) {
        error_log('미디어 업로드 완료 후 처리: 첨부 ID ' . $attachment_id);
        return $metadata;
    }
    
    /**
     * srcset 속성 추가 함수
     */
    public function add_srcset($html, $post_id, $post_thumbnail_id, $size, $attr) {
        // 관리자 설정 옵션 가져오기
        $options = get_option('img_manage_options');
        
        // srcset 기능이 비활성화되어 있으면 원본 HTML 반환
        if (!isset($options['enable_srcset']) || intval($options['enable_srcset']) !== 1) {
            return $html;
        }
        
        // 이미지 정보 가져오기
        $image = wp_get_attachment_metadata($post_thumbnail_id);
        
        // 이미지 메타데이터가 없으면 원본 HTML 반환
        if (empty($image) || !isset($image['width']) || !isset($image['height'])) {
            return $html;
        }
        
        // srcset 속성이 이미 있는지 확인
        if (strpos($html, 'srcset=') !== false) {
            return $html;
        }
        
        // 이미지 URL 가져오기
        $image_src = wp_get_attachment_image_src($post_thumbnail_id, $size);
        if (!$image_src) {
            return $html;
        }
        
        // 다양한 크기의 이미지 URL 수집
        $srcset = array();
        
        // 원본 이미지
        $srcset[] = $image_src[0] . ' ' . $image_src[1] . 'w';
        
        // 중간 크기 이미지 (있는 경우)
        $medium_width = isset($options['medium_image_width']) ? intval($options['medium_image_width']) : 1200;
        $medium_src = wp_get_attachment_image_src($post_thumbnail_id, 'medium');
        if ($medium_src && $medium_src[1] <= $medium_width) {
            $srcset[] = $medium_src[0] . ' ' . $medium_src[1] . 'w';
        }
        
        // 작은 크기 이미지 (있는 경우)
        $small_width = isset($options['small_image_width']) ? intval($options['small_image_width']) : 768;
        $small_src = wp_get_attachment_image_src($post_thumbnail_id, 'thumbnail');
        if ($small_src && $small_src[1] <= $small_width) {
            $srcset[] = $small_src[0] . ' ' . $small_src[1] . 'w';
        }
        
        // srcset 속성이 있는 경우 HTML에 추가
        if (!empty($srcset)) {
            $srcset_attr = 'srcset="' . implode(', ', $srcset) . '"';
            $sizes_attr = 'sizes="(max-width: ' . $small_width . 'px) 100vw, (max-width: ' . $medium_width . 'px) 50vw, ' . $image_src[1] . 'px"';
            
            // img 태그에 srcset과 sizes 속성 추가
            $html = str_replace('<img ', '<img ' . $srcset_attr . ' ' . $sizes_attr . ' ', $html);
        }
        
        return $html;
    }
    
    /**
     * 이미지 alt 태그 개선 함수
     */
    public function improve_alt_tags($post_id) {
        // 자동 저장이나 리비전인 경우 건너뛰기
        if (wp_is_post_revision($post_id) || wp_is_post_autosave($post_id)) {
            return;
        }
        
        // 포스트 제목 가져오기
        $post_title = get_the_title($post_id);
        
        // 포스트에 첨부된 이미지 가져오기
        $attachments = get_attached_media('image', $post_id);
        
        foreach ($attachments as $attachment) {
            // 이미지 파일명에서 확장자와 타임스탬프/해시 제거
            $file_name = pathinfo($attachment->post_title, PATHINFO_FILENAME);
            $file_name = preg_replace('/-\d{10}-[a-f0-9]{6}$/', '', $file_name);
            
            // 본문 이미지용 alt 태그 생성 (더 상세한 설명)
            $content_alt = $post_title . ' - ' . $file_name;
            
            // alt 태그 업데이트
            update_post_meta($attachment->ID, '_wp_attachment_image_alt', $content_alt);
        }
    }
    
    /**
     * 이미지가 업로드될 때 alt 태그 설정
     */
    public function set_alt_on_upload($attachment_id) {
        // 첨부 파일 정보 가져오기
        $attachment = get_post($attachment_id);
        
        // 첨부된 포스트 ID 가져오기
        $parent_id = $attachment->post_parent;
        
        if ($parent_id) {
            // 부모 포스트 제목 가져오기
            $post_title = get_the_title($parent_id);
            
            // 이미지 파일명에서 확장자와 타임스탬프/해시 제거
            $file_name = pathinfo($attachment->post_title, PATHINFO_FILENAME);
            $file_name = preg_replace('/-\d{10}-[a-f0-9]{6}$/', '', $file_name);
            
            // 본문 이미지용 alt 태그 생성 (더 상세한 설명)
            $content_alt = $post_title . ' - ' . $file_name;
            
            // alt 태그 업데이트
            update_post_meta($attachment_id, '_wp_attachment_image_alt', $content_alt);
        } else {
            // 부모 포스트가 없는 경우 (미디어 라이브러리에 직접 업로드)
            // 이미지 파일명에서 확장자와 타임스탬프/해시 제거
            $file_name = pathinfo($attachment->post_title, PATHINFO_FILENAME);
            $file_name = preg_replace('/-\d{10}-[a-f0-9]{6}$/', '', $file_name);
            
            // 의미 있는 alt 태그 생성
            $new_alt = '이미지 - ' . $file_name;
            
            // alt 태그 업데이트
            update_post_meta($attachment_id, '_wp_attachment_image_alt', $new_alt);
        }
    }
    
    /**
     * 본문의 이미지 태그에서 alt 태그 처리 (사용자가 설정한 alt 태그 존중)
     */
    public function title_to_alt($content) {
        // 정규 표현식으로 img 태그를 찾음
        $pattern = '/<img[^>]*>/i';
        
        // 콜백 함수로 각 img 태그 처리
        $content = preg_replace_callback($pattern, function($matches) {
            $img_tag = $matches[0];
            
            // title 속성 확인
            if (preg_match('/title=["\']([^"\']*)["\']/', $img_tag, $title_matches)) {
                $title = $title_matches[1];
                
                // alt 속성 확인
                if (preg_match('/alt=["\']([^"\']*)["\']/', $img_tag, $alt_matches)) {
                    $alt = $alt_matches[1];
                    
                    // alt가 비어있거나 자동 생성된 형식(포스트 제목 - 파일명)이면 title로 교체
                    // 사용자가 직접 설정한 alt는 유지
                    if (empty($alt) || preg_match('/^.*\s-\s.*$/', $alt)) {
                        $img_tag = preg_replace('/alt=["\'][^"\']*["\']/i', 'alt="' . esc_attr($title) . '"', $img_tag);
                    }
                } else {
                    // alt 속성이 없으면 추가
                    $img_tag = str_replace('<img', '<img alt="' . esc_attr($title) . '"', $img_tag);
                }
            }
            
            return $img_tag;
        }, $content);
        
        return $content;
    }
    
    /**
     * 썸네일의 alt 태그 처리 (더 일반적인 설명)
     */
    public function filter_thumbnail_alt($html, $post_id, $post_thumbnail_id, $size, $attr) {
        // 포스트 제목 가져오기
        $post_title = get_the_title($post_id);
        
        // 썸네일용 alt 태그 생성 (더 일반적인 설명)
        $thumbnail_alt = $post_title;
        
        // alt 속성 찾기 및 교체
        if (preg_match('/alt=["\'](.*?)["\']/i', $html, $alt_match)) {
            $html = str_replace($alt_match[0], 'alt="' . esc_attr($thumbnail_alt) . '"', $html);
        }
        
        return $html;
    }
    
    /**
     * 커스텀 썸네일 클래스가 있는 이미지의 alt 태그 처리
     */
    public function custom_thumbnail_alt($content) {
        global $post;
        if (!$post) return $content;
        
        $post_title = get_the_title($post->ID);
        $thumbnail_alt = $post_title;
        
        // custom-thumb 클래스가 있는 이미지 태그 찾기
        if (preg_match_all('/<img[^>]*class=["\'][^"\']*custom-thumb[^"\']*["\'][^>]*>/i', $content, $matches)) {
            foreach ($matches[0] as $img_tag) {
                // alt 속성 찾기 및 교체
                if (preg_match('/alt=["\'](.*?)["\']/i', $img_tag, $alt_match)) {
                    $new_img_tag = str_replace($alt_match[0], 'alt="' . esc_attr($thumbnail_alt) . '"', $img_tag);
                    $content = str_replace($img_tag, $new_img_tag, $content);
                }
            }
        }
        
        return $content;
    }
    
    /**
     * Pixabay 이미지 변환 함수
     * 이 함수는 Pixabay API를 통해 가져온 이미지를 변환하기 위해 사용됩니다.
     */
    public function convert_pixabay_image($file_path) {
        // 이미지 변환 함수와 동일한 로직을 사용하되 Pixabay 이미지에 맞게 조정
        $upload = array(
            'file' => $file_path,
            'url' => '',  // 변환 함수에서는 URL을 사용하지 않음
            'type' => mime_content_type($file_path)
        );
        
        // 변환 함수 호출
        $converted = $this->convert_image_to_target($upload);
        
        return $converted;
    }
}