<?php
/*
 * 파일명: class-core-web-vitals.php
 * 위치: /wp-content/plugins/image-format-converter/includes/
 * 기능: Core Web Vitals 최적화 기능 구현 (이미지 지연 로딩, 사이즈 명시, LCP 최적화)
 * 작성일: 2025-04-21
 */

// 직접 접근 방지
if (!defined('ABSPATH')) {
    exit;
}

class Core_Web_Vitals {
    /**
     * 클래스 생성자
     */
    public function __construct() {
        // 설정 가져오기
        $options = get_option('img_manage_options');
        
        // 이미지 지연 로딩 기능
        if (isset($options['lazy_loading']) && $options['lazy_loading'] == 1) {
            add_filter('wp_get_attachment_image_attributes', array($this, 'add_lazy_loading'), 10, 3);
            add_filter('the_content', array($this, 'add_lazy_loading_to_content_images'), 20);
        }
        
        // 이미지 치수 명시 기능
        if (isset($options['add_dimensions']) && $options['add_dimensions'] == 1) {
            add_filter('wp_get_attachment_image_attributes', array($this, 'add_image_dimensions'), 10, 3);
            add_filter('the_content', array($this, 'add_dimensions_to_content_images'), 20);
        }
        
        // LCP 최적화 기능
        if (isset($options['prioritize_lcp']) && $options['prioritize_lcp'] == 1) {
            add_filter('wp_get_attachment_image_attributes', array($this, 'prioritize_lcp_images'), 10, 3);
            add_filter('the_content', array($this, 'prioritize_lcp_content_images'), 20);
            add_action('wp_head', array($this, 'add_preload_for_featured_image'), 1);
        }
        
        // 공통 스타일 및 스크립트 추가
        add_action('wp_footer', array($this, 'add_intersection_observer_script'), 99);
        
        // 이미지 버퍼링 기능
        add_filter('the_content', array($this, 'buffer_images_for_lcp'), 10);
    }
    
    /**
     * 이미지에 지연 로딩 추가
     */
    public function add_lazy_loading($attr, $attachment, $size) {
        // 이미 loading 속성이 있으면 수정하지 않음
        if (isset($attr['loading'])) {
            return $attr;
        }
        
        // 이미지 크기 정보 가져오기
        $image_meta = wp_get_attachment_metadata($attachment->ID);
        
        // 워드프레스 기본 지연 로딩 사용
        $attr['loading'] = 'lazy';
        
        // LCP 대상 이미지는 지연 로딩 제외 (중요 이미지)
        if ($this->is_lcp_candidate($attachment->ID, $image_meta)) {
            $attr['loading'] = 'eager';
        }
        
        return $attr;
    }
    
    /**
     * 본문 이미지에 지연 로딩 추가
     */
    public function add_lazy_loading_to_content_images($content) {
        if (empty($content)) {
            return $content;
        }
        
        // DOM 파싱을 위한 DOMDocument 사용
        $dom = new DOMDocument();
        
        // 경고 메시지 억제
        libxml_use_internal_errors(true);
        $dom->loadHTML(mb_convert_encoding($content, 'HTML-ENTITIES', 'UTF-8'));
        libxml_clear_errors();
        
        // 이미지 태그 찾기
        $images = $dom->getElementsByTagName('img');
        
        // 첫 번째 이미지는 LCP 후보로 간주하고 지연 로딩 제외
        $first_image_processed = false;
        
        foreach ($images as $img) {
            // 이미 loading 속성이 있으면 수정하지 않음
            if ($img->hasAttribute('loading')) {
                continue;
            }
            
            // 첫 번째 이미지는 즉시 로드
            if (!$first_image_processed) {
                $img->setAttribute('loading', 'eager');
                $first_image_processed = true;
            } else {
                $img->setAttribute('loading', 'lazy');
            }
        }
        
        // 변경된 HTML 반환
        $body = $dom->getElementsByTagName('body')->item(0);
        $content = '';
        foreach ($body->childNodes as $node) {
            $content .= $dom->saveHTML($node);
        }
        
        return $content;
    }
    
    /**
     * 이미지에 치수 명시
     */
    public function add_image_dimensions($attr, $attachment, $size) {
        // 이미 width와 height 속성이 있으면 수정하지 않음
        if (isset($attr['width']) && isset($attr['height'])) {
            return $attr;
        }
        
        // 이미지 크기 정보 가져오기
        $image_meta = wp_get_attachment_metadata($attachment->ID);
        
        if (is_array($image_meta)) {
            // 지정된 크기의 이미지 치수
            if (isset($image_meta['sizes'][$size]) && isset($image_meta['sizes'][$size]['width']) && isset($image_meta['sizes'][$size]['height'])) {
                $attr['width'] = $image_meta['sizes'][$size]['width'];
                $attr['height'] = $image_meta['sizes'][$size]['height'];
            } 
            // 원본 이미지 치수
            else if (isset($image_meta['width']) && isset($image_meta['height'])) {
                $attr['width'] = $image_meta['width'];
                $attr['height'] = $image_meta['height'];
            }
        }
        
        return $attr;
    }
    
    /**
     * 본문 이미지에 치수 명시
     */
    public function add_dimensions_to_content_images($content) {
        if (empty($content)) {
            return $content;
        }
        
        // DOM 파싱을 위한 DOMDocument 사용
        $dom = new DOMDocument();
        
        // 경고 메시지 억제
        libxml_use_internal_errors(true);
        $dom->loadHTML(mb_convert_encoding($content, 'HTML-ENTITIES', 'UTF-8'));
        libxml_clear_errors();
        
        // 이미지 태그 찾기
        $images = $dom->getElementsByTagName('img');
        
        foreach ($images as $img) {
            // 이미 width와 height 속성이 있으면 수정하지 않음
            if ($img->hasAttribute('width') && $img->hasAttribute('height')) {
                continue;
            }
            
            // 이미지 URL에서 첨부 ID 추출
            $img_src = $img->getAttribute('src');
            $attachment_id = $this->get_attachment_id_from_url($img_src);
            
            if ($attachment_id) {
                // 이미지 메타데이터 가져오기
                $image_meta = wp_get_attachment_metadata($attachment_id);
                
                if (is_array($image_meta) && isset($image_meta['width']) && isset($image_meta['height'])) {
                    // 원본 이미지 치수 설정
                    $img->setAttribute('width', $image_meta['width']);
                    $img->setAttribute('height', $image_meta['height']);
                }
            } else {
                // 첨부 ID를 찾을 수 없는 경우 (외부 이미지 등) naturalWidth/naturalHeight 스크립트 추가
                $img->setAttribute('onload', 'if(!this.hasAttribute("width")){this.setAttribute("width", this.naturalWidth);this.setAttribute("height", this.naturalHeight);}');
            }
        }
        
        // 변경된 HTML 반환
        $body = $dom->getElementsByTagName('body')->item(0);
        $content = '';
        foreach ($body->childNodes as $node) {
            $content .= $dom->saveHTML($node);
        }
        
        return $content;
    }
    
    /**
     * LCP 후보 이미지 최적화
     */
    public function prioritize_lcp_images($attr, $attachment, $size) {
        // 이미지 크기 정보 가져오기
        $image_meta = wp_get_attachment_metadata($attachment->ID);
        
        // LCP 후보 이미지 확인
        if ($this->is_lcp_candidate($attachment->ID, $image_meta)) {
            // 즉시 로드 설정
            $attr['loading'] = 'eager';
            
            // fetchpriority 속성 추가 (Chrome 기준 최적화)
            $attr['fetchpriority'] = 'high';
            
            // 완료 이벤트 추적 (선택 사항)
            $attr['onload'] = 'if(window.performance && window.performance.mark) window.performance.mark("lcp-image-loaded");';
        }
        
        return $attr;
    }
    
    /**
     * 본문 LCP 후보 이미지 최적화
     */
    public function prioritize_lcp_content_images($content) {
        if (empty($content)) {
            return $content;
        }
        
        // DOM 파싱을 위한 DOMDocument 사용
        $dom = new DOMDocument();
        
        // 경고 메시지 억제
        libxml_use_internal_errors(true);
        $dom->loadHTML(mb_convert_encoding($content, 'HTML-ENTITIES', 'UTF-8'));
        libxml_clear_errors();
        
        // 이미지 태그 찾기
        $images = $dom->getElementsByTagName('img');
        
        // 포스트 내용에서 첫 번째 이미지는 LCP 후보로 간주
        $first_image_processed = false;
        
        foreach ($images as $img) {
            // 첫 번째 이미지는 LCP 최적화
            if (!$first_image_processed && $img->getAttribute('width') >= 300) { // 최소 크기 조건 추가
                $img->setAttribute('loading', 'eager');
                $img->setAttribute('fetchpriority', 'high');
                $img->setAttribute('onload', 'if(window.performance && window.performance.mark) window.performance.mark("lcp-content-image-loaded");');
                $first_image_processed = true;
            }
        }
        
        // 변경된 HTML 반환
        $body = $dom->getElementsByTagName('body')->item(0);
        $content = '';
        foreach ($body->childNodes as $node) {
            $content .= $dom->saveHTML($node);
        }
        
        return $content;
    }
    
    /**
     * 대표 이미지 프리로드 추가
     */
    public function add_preload_for_featured_image() {
        // 싱글 포스트에서만 실행
        if (!is_singular()) {
            return;
        }
        
        $post_id = get_the_ID();
        
        // 대표 이미지 ID 가져오기
        $thumbnail_id = get_post_thumbnail_id($post_id);
        
        if (!$thumbnail_id) {
            return;
        }
        
        // 대표 이미지 URL 가져오기
        $thumbnail_src = wp_get_attachment_image_src($thumbnail_id, 'large');
        
        if (!$thumbnail_src) {
            return;
        }
        
        // 프리로드 링크 추가
        echo '<link rel="preload" as="image" href="' . esc_url($thumbnail_src[0]) . '" fetchpriority="high">';
    }
    
    /**
     * URL로부터 첨부 ID 가져오기
     */
    private function get_attachment_id_from_url($url) {
        global $wpdb;
        
        // 첨부 페이지 URL 처리
        $attachment = $wpdb->get_col($wpdb->prepare("SELECT ID FROM $wpdb->posts WHERE guid='%s';", $url));
        
        if (!empty($attachment)) {
            return $attachment[0];
        }
        
        // 업로드 디렉토리 URL 제거
        $upload_dir = wp_upload_dir();
        $url = str_replace($upload_dir['baseurl'] . '/', '', $url);
        
        // 첨부 메타데이터 확인
        $attachment = $wpdb->get_col($wpdb->prepare("SELECT post_id FROM $wpdb->postmeta WHERE meta_key='_wp_attached_file' AND meta_value='%s';", $url));
        
        if (!empty($attachment)) {
            return $attachment[0];
        }
        
        return false;
    }
    
    /**
     * LCP 후보 이미지인지 확인
     */
    private function is_lcp_candidate($attachment_id, $image_meta) {
        // 대표 이미지인지 확인
        $featured_id = get_post_thumbnail_id(get_the_ID());
        
        if ($attachment_id == $featured_id) {
            return true;
        }
        
        // 이미지 크기가 큰 경우 (LCP가 될 가능성이 높음)
        if (is_array($image_meta) && isset($image_meta['width']) && isset($image_meta['height'])) {
            // 500px 이상의 큰 이미지는 LCP 후보로 간주
            if ($image_meta['width'] >= 500 && $image_meta['height'] >= 300) {
                // 포스트에서 이 이미지의 위치 확인 (첫 번째 이미지가 LCP 후보가 될 가능성이 높음)
                $post_id = get_the_ID();
                
                if ($post_id) {
                    $post = get_post($post_id);
                    
                    if ($post && strpos($post->post_content, wp_get_attachment_url($attachment_id)) !== false) {
                        // 포스트 내용에 이 이미지가 포함되어 있고 크기가 크면 LCP 후보
                        return true;
                    }
                }
            }
        }
        
        return false;
    }
    
    /* 본문 이미지 버퍼링으로 LCP 최적화 */
	public function buffer_images_for_lcp($content) {
		if (empty($content) || !is_singular()) {
			return $content;
		}
		
		// 첫 번째 이미지 추출
		if (preg_match('/<img[^>]+src=[\'"]([^\'"]+)[\'"][^>]*>/i', $content, $matches)) {
			$first_image_url = $matches[1];
			
			// 첫 번째 이미지 URL을 전역 변수에 저장 (스크립트에서 사용)
			global $first_image_for_lcp;
			$first_image_for_lcp = $first_image_url;
			
			// 버퍼링 스크립트를 wp_footer에 추가
			add_action('wp_footer', array($this, 'add_lcp_buffer_script'), 1);
		}
		
		return $content;
	}

	/**
	 * LCP 버퍼링 스크립트 추가
	 */
	public function add_lcp_buffer_script() {
		global $first_image_for_lcp;
		
		if (empty($first_image_for_lcp)) {
			return;
		}
		
		?>
		<script>
		(function() {
			// 첫 번째 이미지 프리페치
			var img = new Image();
			img.src = '<?php echo esc_url($first_image_for_lcp); ?>';
			
			// 버퍼링 상태 추적
			img.onload = function() {
				if(window.performance && window.performance.mark) {
					window.performance.mark('first-image-loaded');
				}
			};
		})();
		</script>
		<?php
	}
    
    /**
     * Intersection Observer 스크립트 추가 (추가 최적화)
     */
    public function add_intersection_observer_script() {
        // 싱글 포스트에서만 실행
        if (!is_singular()) {
            return;
        }
        
        ?>
        <script>
        (function() {
            // 이미지가 뷰포트에 진입할 때 로드하는 고급 지연 로딩 구현
            if ('IntersectionObserver' in window) {
                var imageObserver = new IntersectionObserver(function(entries, observer) {
                    entries.forEach(function(entry) {
                        if (entry.isIntersecting) {
                            var image = entry.target;
                            
                            // data-src 속성이 있는 경우 (완전한 지연 로딩)
                            if (image.dataset.src) {
                                image.src = image.dataset.src;
                                delete image.dataset.src;
                            }
                            
                            // data-srcset 속성이 있는 경우
                            if (image.dataset.srcset) {
                                image.srcset = image.dataset.srcset;
                                delete image.dataset.srcset;
                            }
                            
                            // 이미지에 로드 이벤트 추가
                            image.onload = function() {
                                // 로드 후 클래스 제거
                                image.classList.remove('lazy-load');
                                image.classList.add('loaded');
                            };
                            
                            // 관찰 중단
                            observer.unobserve(image);
                        }
                    });
                }, {
                    rootMargin: '200px 0px' // 뷰포트 200px 전에 미리 로드 시작
                });
                
                // lazy-load 클래스를 가진 모든 이미지 관찰
                document.querySelectorAll('img.lazy-load').forEach(function(image) {
                    imageObserver.observe(image);
                });
                
                // LCP 측정 및 보고
                if (window.performance && window.performance.mark) {
                    // LCP 마크 측정 (이미 설정된 마크)
                    window.addEventListener('load', function() {
                        setTimeout(function() {
                            if (performance.getEntriesByName('lcp-image-loaded').length) {
                                console.log('LCP Image loaded successfully');
                            }
                        }, 2000);
                    });
                }
            }
        })();
        </script>
        <?php
    }
}