<?php
/*
 * 파일명: class-admin-settings.php
 * 위치: /wp-content/plugins/image-format-converter/includes/
 * 기능: 플러그인 관리자 설정 페이지
 * 작성일: 2025-04-21
 */

// 직접 접근 방지
if (!defined('ABSPATH')) {
    exit;
}

class Admin_Settings {
    /**
     * 클래스 생성자
     */
    public function __construct() {
        add_action('admin_menu', array($this, 'admin_menu'));
        add_action('admin_init', array($this, 'register_settings'));
        add_action('admin_enqueue_scripts', array($this, 'admin_styles'));
    }
    
    /**
     * 관리자 메뉴 추가
     */
    public function admin_menu() {
        add_menu_page(
            '이미지 관리',
            '이미지 관리',
            'manage_options',
            'image-management',
            array($this, 'options_page'),
            'dashicons-format-image',
            80
        );
    }
    
    /**
     * 관리자 스타일 추가
     */
    public function admin_styles($hook) {
        if ($hook != 'toplevel_page_image-management') {
            return;
        }
        
        wp_enqueue_style('img-manage-admin-style', IMG_MANAGE_URL . 'assets/css/admin-style.css', array(), IMG_MANAGE_VERSION);
        wp_enqueue_script('img-manage-admin-script', IMG_MANAGE_URL . 'assets/js/admin-script.js', array('jquery'), IMG_MANAGE_VERSION, true);
    }
    
    /**
     * 설정 등록
     */
    public function register_settings() {
        // 설정 등록 시 sanitize 콜백 함수 지정
        register_setting(
            'img_manage_options_group',  // 옵션 그룹
            'img_manage_options',        // 옵션 이름
            array($this, 'sanitize_options') // 유효성 검사 및 정리 콜백 함수
        );
        
        // 설정 저장 전 기본값 확인 및 설정
        add_filter('pre_update_option_img_manage_options', function($new_value, $old_value) {
            $defaults = img_manage_default_options();
            
            // 누락된 설정이 있는지 확인하고 기본값으로 채우기
            foreach ($defaults as $key => $default_value) {
                if (!isset($new_value[$key])) {
                    $new_value[$key] = $default_value;
                }
            }
            
            return $new_value;
        }, 10, 2);

        // 기본 섹션
        add_settings_section(
            'img_manage_main_section',
            '이미지 기본 설정',
            array($this, 'section_callback'),
            'image-management'
        );

        // 이미지 크기 설정
        add_settings_field(
            'image_sizes',
            '이미지 크기 설정',
            array($this, 'image_sizes_callback'),
            'image-management',
            'img_manage_main_section'
        );

        // 지정된 이미지보다 작으면 변환하지 않음
        add_settings_field(
            'skip_small_images',
            '작은 이미지 처리',
            array($this, 'skip_small_callback'),
            'image-management',
            'img_manage_main_section'
        );

        // SEO 설정 섹션
        add_settings_section(
            'img_manage_seo_section',
            'SEO 최적화 설정',
            array($this, 'seo_section_callback'),
            'image-management'
        );

        // 파일명 관리
        add_settings_field(
            'seo_filename',
            '파일명 관리',
            array($this, 'seo_filename_callback'),
            'image-management',
            'img_manage_seo_section'
        );

        // 메타데이터 보존
        add_settings_field(
            'preserve_metadata',
            '메타데이터 보존',
            array($this, 'preserve_metadata_callback'),
            'image-management',
            'img_manage_seo_section'
        );

        // 반응형 이미지 섹션
        add_settings_section(
            'img_manage_responsive_section',
            '반응형 이미지 설정',
            array($this, 'responsive_section_callback'),
            'image-management'
        );

        // srcset 지원
        add_settings_field(
            'enable_srcset',
            'srcset 지원',
            array($this, 'enable_srcset_callback'),
            'image-management',
            'img_manage_responsive_section'
        );
        
        // Core Web Vitals 섹션 추가
        add_settings_section(
            'img_manage_web_vitals_section',
            'Core Web Vitals 최적화',
            array($this, 'web_vitals_section_callback'),
            'image-management'
        );
        
        // 지연 로딩 설정
        add_settings_field(
            'lazy_loading',
            '이미지 지연 로딩',
            array($this, 'lazy_loading_callback'),
            'image-management',
            'img_manage_web_vitals_section'
        );
        
        // 이미지 치수 명시
        add_settings_field(
            'add_dimensions',
            '이미지 치수 명시',
            array($this, 'add_dimensions_callback'),
            'image-management',
            'img_manage_web_vitals_section'
        );
        
        // LCP 최적화
        add_settings_field(
            'prioritize_lcp',
            'LCP 최적화',
            array($this, 'prioritize_lcp_callback'),
            'image-management',
            'img_manage_web_vitals_section'
        );

        // 이미지 변환 설정 섹션
        add_settings_section(
            'img_manage_conversion_section',
            '이미지 변환 설정',
            array($this, 'conversion_section_callback'),
            'image-management'
        );

        // PNG 변환 옵션
        add_settings_field(
            'png_conversion',
            'PNG 변환 옵션',
            array($this, 'png_conversion_callback'),
            'image-management',
            'img_manage_conversion_section'
        );

        // JPG 변환 옵션
        add_settings_field(
            'jpg_conversion',
            'JPG 변환 옵션',
            array($this, 'jpg_conversion_callback'),
            'image-management',
            'img_manage_conversion_section'
        );

        // WEBP 변환 옵션
        add_settings_field(
            'webp_conversion',
            'WEBP 변환 옵션',
            array($this, 'webp_conversion_callback'),
            'image-management',
            'img_manage_conversion_section'
        );
        
        // Pixabay API 섹션 추가
        add_settings_section(
            'img_manage_pixabay_section',
            'Pixabay API 설정',
            array($this, 'pixabay_section_callback'),
            'image-management'
        );
        
        // Pixabay 활성화 설정
        add_settings_field(
            'pixabay_enabled',
            'Pixabay 통합',
            array($this, 'pixabay_enabled_callback'),
            'image-management',
            'img_manage_pixabay_section'
        );
        
        // Pixabay 기본 설정
        add_settings_field(
            'pixabay_defaults',
            'Pixabay 기본 설정',
            array($this, 'pixabay_defaults_callback'),
            'image-management',
            'img_manage_pixabay_section'
        );
    }
    
    /**
     * 설정 저장 전 체크박스 값 처리
     */
    public function sanitize_options($new_options) {
        $defaults = img_manage_default_options();
        
        // 체크박스 필드 목록
        $checkbox_fields = array(
            'png_enabled',
            'jpg_enabled', 
            'webp_enabled',
            'skip_small_images',
            'preserve_metadata',
            'enable_srcset',
            'seo_filename',
            'lazy_loading',
            'add_dimensions',
            'prioritize_lcp',
            'pixabay_enabled'
        );
        
        // 체크박스가 체크되지 않았을 때 값이 넘어오지 않으므로,
        // 명시적으로 0으로 설정
        foreach ($checkbox_fields as $field) {
            if (!isset($new_options[$field])) {
                $new_options[$field] = 0;
            }
        }
        
        // 품질 설정 값 검증
        $quality_fields = array('png_quality', 'jpg_quality', 'webp_quality');
        foreach ($quality_fields as $field) {
            if (!isset($new_options[$field]) || !is_numeric($new_options[$field]) || $new_options[$field] < 10 || $new_options[$field] > 100) {
                $new_options[$field] = $defaults[$field];
            } else {
                $new_options[$field] = intval($new_options[$field]);
            }
        }
        
        // Pixabay 설정 값 검증
        if (isset($new_options['pixabay_per_page']) && (!is_numeric($new_options['pixabay_per_page']) || $new_options['pixabay_per_page'] < 3 || $new_options['pixabay_per_page'] > 200)) {
            $new_options['pixabay_per_page'] = $defaults['pixabay_per_page'];
        } else if (isset($new_options['pixabay_per_page'])) {
            $new_options['pixabay_per_page'] = intval($new_options['pixabay_per_page']);
        }
        
        return $new_options;
    }
    
    /**
     * 섹션 콜백 함수들
     */
    public function section_callback() {
        echo '<p>이미지 업로드 시 적용되는 기본 설정입니다.</p>';
    }

    public function seo_section_callback() {
        echo '<p>SEO 최적화를 위한 이미지 설정입니다.</p>';
    }

    public function responsive_section_callback() {
        echo '<p>다양한 디바이스에 최적화된 반응형 이미지 제공을 위한 설정입니다.</p>';
    }

    public function conversion_section_callback() {
        echo '<p>이미지 형식별 변환 및 압축 설정입니다. WEBP는 최신 PHP 버전에서 지원됩니다.</p>';
    }
    
    public function web_vitals_section_callback() {
        echo '<p>Core Web Vitals 점수 향상을 위한 이미지 최적화 설정입니다. 페이지 속도와 사용자 경험을 개선합니다.</p>';
    }
    
    public function pixabay_section_callback() {
        echo '<p>Pixabay API를 통한 무료 이미지 검색 및 삽입 기능 설정입니다. 글 작성 시 무료 이미지를 쉽게 추가할 수 있습니다.</p>';
    }
    
    /**
     * 이미지 크기 설정 콜백 함수
     */
    public function image_sizes_callback() {
        $options = get_option('img_manage_options');
        $max_width = isset($options['max_image_width']) && !empty($options['max_image_width']) ? $options['max_image_width'] : 1920;
        $medium_width = isset($options['medium_image_width']) && !empty($options['medium_image_width']) ? $options['medium_image_width'] : 1200;
        $small_width = isset($options['small_image_width']) && !empty($options['small_image_width']) ? $options['small_image_width'] : 768;
        $thumbnail_width = isset($options['thumbnail_width']) && !empty($options['thumbnail_width']) ? $options['thumbnail_width'] : 400;
        
        echo '<div class="img-option-row">';
        echo '<label for="max-width">최대 가로크기 (픽셀):</label> ';
        echo '<input type="number" id="max-width" name="img_manage_options[max_image_width]" value="' . esc_attr($max_width) . '" class="img-number" min="0" required>';
        echo '<span class="img-desc">원본 이미지의 최대 가로 크기입니다. 권장값: 1920</span>';
        echo '</div>';
        
        echo '<div class="img-option-row">';
        echo '<label for="medium-width">중간 가로크기 (픽셀):</label> ';
        echo '<input type="number" id="medium-width" name="img_manage_options[medium_image_width]" value="' . esc_attr($medium_width) . '" class="img-number" min="0" required>';
        echo '<span class="img-desc">태블릿 등 중간 크기 디바이스용 이미지입니다. 권장값: 1200</span>';
        echo '</div>';
        
        echo '<div class="img-option-row">';
        echo '<label for="small-width">작은 가로크기 (픽셀):</label> ';
        echo '<input type="number" id="small-width" name="img_manage_options[small_image_width]" value="' . esc_attr($small_width) . '" class="img-number" min="0" required>';
        echo '<span class="img-desc">모바일 디바이스용 이미지입니다. 권장값: 768</span>';
        echo '</div>';
        
        echo '<div class="img-option-row">';
        echo '<label for="thumbnail-width">썸네일 가로크기 (픽셀):</label> ';
        echo '<input type="number" id="thumbnail-width" name="img_manage_options[thumbnail_width]" value="' . esc_attr($thumbnail_width) . '" class="img-number" min="0" required>';
        echo '<span class="img-desc">미리보기용 썸네일 이미지입니다. 권장값: 400</span>';
        echo '</div>';
    }
    
    /**
     * 작은 이미지 처리 콜백 함수
     */
    public function skip_small_callback() {
        $options = get_option('img_manage_options');
        $skip_small = isset($options['skip_small_images']) ? (int)$options['skip_small_images'] : 1;
        
        echo '<div class="img-option-row">';
        echo '<input type="checkbox" id="skip-small" name="img_manage_options[skip_small_images]" value="1" ' . checked(1, $skip_small, false) . ' /> ';
        echo '<label for="skip-small">작은 이미지 최적화</label>';
        echo '<span class="img-desc">이미지 크기가 지정한 최대 가로크기보다 작으면 <strong>리사이즈하지 않고 형식만 변환</strong>합니다. 작은 이미지의 품질을 유지하는 데 도움이 됩니다.</span>';
        echo '</div>';
    }
    
    /**
     * SEO 파일명 설정 콜백 함수
     */
    public function seo_filename_callback() {
        $options = get_option('img_manage_options');
        $seo_filename = isset($options['seo_filename']) ? (int)$options['seo_filename'] : 1;
        
        echo '<div class="img-option-row">';
        echo '<input type="checkbox" id="seo-filename" name="img_manage_options[seo_filename]" value="1" ' . checked(1, $seo_filename, false) . ' /> ';
        echo '<label for="seo-filename">SEO 친화적 파일명 사용</label>';
        echo '<span class="img-desc">SEO에 유리한 방식으로 파일명을 관리합니다. 원본 파일명을 최대한 유지하면서 고유성을 보장합니다.</span>';
        echo '</div>';
    }
    
    /**
     * 메타데이터 보존 설정 콜백 함수
     */
    public function preserve_metadata_callback() {
        $options = get_option('img_manage_options');
        $preserve_metadata = isset($options['preserve_metadata']) ? (int)$options['preserve_metadata'] : 1;
        
        echo '<div class="img-option-row">';
        echo '<input type="checkbox" id="preserve-metadata" name="img_manage_options[preserve_metadata]" value="1" ' . checked(1, $preserve_metadata, false) . ' /> ';
        echo '<label for="preserve-metadata">메타데이터 보존</label>';
        echo '<span class="img-desc">이미지의 EXIF, IPTC 등 중요한 메타데이터를 유지합니다.</span>';
        echo '</div>';
    }
    
    /**
     * srcset 지원 설정 콜백 함수
     */
    public function enable_srcset_callback() {
        $options = get_option('img_manage_options');
        $enable_srcset = isset($options['enable_srcset']) ? (int)$options['enable_srcset'] : 1;
        
        echo '<div class="img-option-row">';
        echo '<input type="checkbox" id="enable-srcset" name="img_manage_options[enable_srcset]" value="1" ' . checked(1, $enable_srcset, false) . ' /> ';
        echo '<label for="enable-srcset">srcset 속성 활성화</label>';
        echo '<span class="img-desc">다양한 화면 크기에 최적화된 이미지를 제공하는 srcset 속성을 활성화합니다.</span>';
        echo '</div>';
    }
    
    /**
     * 지연 로딩 설정 콜백 함수
     */
    public function lazy_loading_callback() {
        $options = get_option('img_manage_options');
        $lazy_loading = isset($options['lazy_loading']) ? (int)$options['lazy_loading'] : 1;
        
        echo '<div class="img-option-row">';
        echo '<input type="checkbox" id="lazy-loading" name="img_manage_options[lazy_loading]" value="1" ' . checked(1, $lazy_loading, false) . ' /> ';
        echo '<label for="lazy-loading">이미지 지연 로딩 활성화</label>';
        echo '<span class="img-desc">화면에 보이는 부분의 이미지만 먼저 로드하고 나머지는 스크롤 시 로드합니다. 페이지 로딩 속도를 향상시킵니다.</span>';
        echo '</div>';
    }
    
    /**
     * 이미지 치수 명시 설정 콜백 함수
     */
    public function add_dimensions_callback() {
        $options = get_option('img_manage_options');
        $add_dimensions = isset($options['add_dimensions']) ? (int)$options['add_dimensions'] : 1;
        
        echo '<div class="img-option-row">';
        echo '<input type="checkbox" id="add-dimensions" name="img_manage_options[add_dimensions]" value="1" ' . checked(1, $add_dimensions, false) . ' /> ';
        echo '<label for="add-dimensions">이미지 치수 명시</label>';
        echo '<span class="img-desc">이미지 HTML에 width/height 속성을 명시하여 CLS(Cumulative Layout Shift) 점수를 개선합니다.</span>';
        echo '</div>';
    }
    
    /**
     * LCP 최적화 설정 콜백 함수
     */
    public function prioritize_lcp_callback() {
        $options = get_option('img_manage_options');
        $prioritize_lcp = isset($options['prioritize_lcp']) ? (int)$options['prioritize_lcp'] : 1;
        
        echo '<div class="img-option-row">';
        echo '<input type="checkbox" id="prioritize-lcp" name="img_manage_options[prioritize_lcp]" value="1" ' . checked(1, $prioritize_lcp, false) . ' /> ';
        echo '<label for="prioritize-lcp">LCP 이미지 최적화</label>';
        echo '<span class="img-desc">가장 큰 콘텐츠 요소(LCP)로 식별된 이미지를 우선적으로 로드하여 Core Web Vitals 점수를 향상시킵니다.</span>';
        echo '</div>';
    }
    
    /**
     * PNG 변환 옵션 콜백 함수
     */
    public function png_conversion_callback() {
        $options = get_option('img_manage_options');
        
        // 체크박스 상태
        $enabled = isset($options['png_enabled']) ? (int)$options['png_enabled'] : 1;
        
        $target = isset($options['png_target']) ? $options['png_target'] : 'webp';
        
        // 기본값과 저장된 값을 분리하여 처리
        $default_quality = 80; // 기본값
        $saved_quality = isset($options['png_quality']) && is_numeric($options['png_quality']) && $options['png_quality'] > 0
            ? intval($options['png_quality']) 
            : $default_quality;
        
        echo '<div class="img-format-section png-section">';
        echo '<h3 class="img-format-title">PNG 변환 옵션</h3>';
        
        echo '<div class="img-option-row">';
        echo '<input type="checkbox" id="png-enabled" name="img_manage_options[png_enabled]" value="1" ' . checked(1, $enabled, false) . ' /> ';
        echo '<label for="png-enabled">PNG 변환 활성화</label>';
        echo '</div>';
        
        echo '<div class="img-option-row">';
        echo '<label for="png-target">변환 형식:</label> ';
        echo '<select id="png-target" name="img_manage_options[png_target]" class="img-select">';
        echo '<option value="webp" ' . selected($target, 'webp', false) . '>WEBP (권장)</option>';
        echo '<option value="jpeg" ' . selected($target, 'jpeg', false) . '>JPEG</option>';
        echo '</select>';
        echo '<span class="img-desc">WebP는 PNG보다 파일 크기가 작으면서도 투명도를 지원합니다.</span>';
        echo '</div>';
        
        echo '<div class="img-option-row">';
        echo '<label for="png-quality">압축률 (10~100):</label> ';
        echo '<input type="number" id="png-quality" name="img_manage_options[png_quality]" value="' . esc_attr($saved_quality) . '" class="img-number" min="10" max="100" step="5">';
        echo '<span class="img-desc">높을수록 품질이 좋지만 파일 크기가 커집니다. 80-85가 권장됩니다.</span>';
        echo '</div>';
        
        echo '</div>'; // .img-format-section 종료
    }
    
    /**
     * JPG 변환 옵션 콜백 함수
     */
    public function jpg_conversion_callback() {
        $options = get_option('img_manage_options');
        
        // 체크박스 상태
        $enabled = isset($options['jpg_enabled']) ? (int)$options['jpg_enabled'] : 1;
        
        // 기본값과 저장된 값을 분리하여 처리
        $default_quality = 85; // 기본값
        $saved_quality = isset($options['jpg_quality']) && is_numeric($options['jpg_quality']) && $options['jpg_quality'] > 0
            ? intval($options['jpg_quality']) 
            : $default_quality;
        
        echo '<div class="img-format-section jpg-section">';
        echo '<h3 class="img-format-title">JPG 변환 옵션</h3>';
        
        echo '<div class="img-option-row">';
        echo '<input type="checkbox" id="jpg-enabled" name="img_manage_options[jpg_enabled]" value="1" ' . checked(1, $enabled, false) . ' /> ';
        echo '<label for="jpg-enabled">JPG -> WEBP 변환 활성화</label>';
        echo '<span class="img-desc">JPG는 항상 WEBP로 변환됩니다.</span>';
        echo '</div>';
        
        echo '<div class="img-option-row">';
        echo '<label for="jpg-quality">압축률 (10~100):</label> ';
        echo '<input type="number" id="jpg-quality" name="img_manage_options[jpg_quality]" value="' . esc_attr($saved_quality) . '" class="img-number" min="10" max="100" step="5">';
        echo '<span class="img-desc">높을수록 품질이 좋지만 파일 크기가 커집니다. 85가 권장됩니다.</span>';
        echo '</div>';
        
        echo '</div>'; // .img-format-section 종료
    }
    
    /**
     * WEBP 변환 옵션 콜백 함수
     */
    public function webp_conversion_callback() {
        $options = get_option('img_manage_options');
        
        // 체크박스 상태
        $enabled = isset($options['webp_enabled']) ? (int)$options['webp_enabled'] : 0;
        
        // 기본값과 저장된 값을 분리하여 처리
        $default_quality = 85; // 기본값
        $saved_quality = isset($options['webp_quality']) && is_numeric($options['webp_quality']) && $options['webp_quality'] > 0
            ? intval($options['webp_quality']) 
            : $default_quality;
        
        // 고급 설정으로 레이블 변경 및 설명 추가
        echo '<div class="img-format-section webp-section">';
        echo '<h3 class="img-format-title">WEBP 변환 옵션 (고급 설정)</h3>';
        
        echo '<div class="img-option-row">';
        echo '<input type="checkbox" id="webp-enabled" name="img_manage_options[webp_enabled]" value="1" ' . checked(1, $enabled, false) . ' /> ';
        echo '<label for="webp-enabled">WEBP -> JPEG 변환 활성화</label>';
        echo '<span class="img-desc">일반적으로 WEBP는 이미 최적화된 형식이므로 변환이 필요하지 않습니다. 특수한 경우에만 사용하세요.</span>';
        echo '</div>';
        
        echo '<div class="img-option-row">';
        echo '<label for="webp-quality">압축률 (10~100):</label> ';
        echo '<input type="number" id="webp-quality" name="img_manage_options[webp_quality]" value="' . esc_attr($saved_quality) . '" class="img-number" min="10" max="100" step="5">';
        echo '<span class="img-desc">높을수록 품질이 좋지만 파일 크기가 커집니다.</span>';
        echo '</div>';
        
        echo '</div>'; // .img-format-section 종료
    }
    
    /**
	 * Pixabay 활성화 설정 콜백 함수
	 */
	public function pixabay_enabled_callback() {
		$options = get_option('img_manage_options');
		$pixabay_enabled = isset($options['pixabay_enabled']) ? (int)$options['pixabay_enabled'] : 1;
		$pixabay_api_key = isset($options['pixabay_api_key']) ? $options['pixabay_api_key'] : '';
		
		echo '<div class="img-option-row">';
		echo '<input type="checkbox" id="pixabay-enabled" name="img_manage_options[pixabay_enabled]" value="1" ' . checked(1, $pixabay_enabled, false) . ' /> ';
		echo '<label for="pixabay-enabled">Pixabay 통합 활성화</label>';
		echo '<span class="img-desc">글 편집 화면에서 Pixabay의 무료 이미지를 검색하고 삽입할 수 있습니다.</span>';
		echo '</div>';
		
		echo '<div class="img-option-row">';
		echo '<label for="pixabay-api-key">API 키:</label> ';
		echo '<input type="text" id="pixabay-api-key" name="img_manage_options[pixabay_api_key]" value="' . esc_attr($pixabay_api_key) . '" class="regular-text" />';
		echo '<span class="img-desc">Pixabay API 키를 입력하세요. API 키는 <a href="https://pixabay.com/api/docs/" target="_blank">Pixabay API 문서</a>에서 확인할 수 있습니다.</span>';
		echo '</div>';
	}
    
    /**
     * Pixabay 기본 설정 콜백 함수
     */
    public function pixabay_defaults_callback() {
        $options = get_option('img_manage_options');
        
        // 기본 이미지 타입
        $default_type = isset($options['pixabay_default_type']) ? $options['pixabay_default_type'] : 'photo';
        
        // 기본 방향
        $default_orientation = isset($options['pixabay_default_orientation']) ? $options['pixabay_default_orientation'] : 'all';
        
        // 기본 카테고리
        $default_category = isset($options['pixabay_default_category']) ? $options['pixabay_default_category'] : '';
        
        // 페이지당 결과 수
        $per_page = isset($options['pixabay_per_page']) && is_numeric($options['pixabay_per_page']) ? intval($options['pixabay_per_page']) : 20;
        
        echo '<div class="img-option-row">';
        echo '<label for="pixabay-default-type">기본 이미지 타입:</label> ';
        echo '<select id="pixabay-default-type" name="img_manage_options[pixabay_default_type]" class="img-select">';
        echo '<option value="photo" ' . selected($default_type, 'photo', false) . '>사진</option>';
        echo '<option value="illustration" ' . selected($default_type, 'illustration', false) . '>일러스트레이션</option>';
        echo '<option value="vector" ' . selected($default_type, 'vector', false) . '>벡터 그래픽</option>';
        echo '<option value="all" ' . selected($default_type, 'all', false) . '>모든 이미지</option>';
        echo '</select>';
        echo '<span class="img-desc">검색 시 기본 이미지 타입을 설정합니다.</span>';
        echo '</div>';
        
        echo '<div class="img-option-row">';
        echo '<label for="pixabay-default-orientation">기본 방향:</label> ';
        echo '<select id="pixabay-default-orientation" name="img_manage_options[pixabay_default_orientation]" class="img-select">';
        echo '<option value="all" ' . selected($default_orientation, 'all', false) . '>모든 방향</option>';
        echo '<option value="horizontal" ' . selected($default_orientation, 'horizontal', false) . '>가로 방향</option>';
        echo '<option value="vertical" ' . selected($default_orientation, 'vertical', false) . '>세로 방향</option>';
        echo '</select>';
        echo '<span class="img-desc">검색 시 기본 이미지 방향을 설정합니다.</span>';
        echo '</div>';
        
        echo '<div class="img-option-row">';
        echo '<label for="pixabay-default-category">기본 카테고리:</label> ';
        echo '<select id="pixabay-default-category" name="img_manage_options[pixabay_default_category]" class="img-select">';
        echo '<option value="" ' . selected($default_category, '', false) . '>카테고리 없음</option>';
        echo '<option value="backgrounds" ' . selected($default_category, 'backgrounds', false) . '>배경</option>';
        echo '<option value="fashion" ' . selected($default_category, 'fashion', false) . '>패션</option>';
        echo '<option value="nature" ' . selected($default_category, 'nature', false) . '>자연</option>';
        echo '<option value="science" ' . selected($default_category, 'science', false) . '>과학</option>';
        echo '<option value="education" ' . selected($default_category, 'education', false) . '>교육</option>';
        echo '<option value="health" ' . selected($default_category, 'health', false) . '>건강</option>';
        echo '<option value="people" ' . selected($default_category, 'people', false) . '>사람</option>';
        echo '<option value="religion" ' . selected($default_category, 'religion', false) . '>종교</option>';
        echo '<option value="places" ' . selected($default_category, 'places', false) . '>장소</option>';
        echo '<option value="animals" ' . selected($default_category, 'animals', false) . '>동물</option>';
        echo '<option value="industry" ' . selected($default_category, 'industry', false) . '>산업</option>';
        echo '<option value="computer" ' . selected($default_category, 'computer', false) . '>컴퓨터</option>';
        echo '<option value="food" ' . selected($default_category, 'food', false) . '>음식</option>';
        echo '<option value="sports" ' . selected($default_category, 'sports', false) . '>스포츠</option>';
        echo '<option value="transportation" ' . selected($default_category, 'transportation', false) . '>교통</option>';
        echo '<option value="travel" ' . selected($default_category, 'travel', false) . '>여행</option>';
        echo '<option value="buildings" ' . selected($default_category, 'buildings', false) . '>건물</option>';
        echo '<option value="business" ' . selected($default_category, 'business', false) . '>비즈니스</option>';
        echo '<option value="music" ' . selected($default_category, 'music', false) . '>음악</option>';
        echo '</select>';
        echo '<span class="img-desc">검색 시 기본 카테고리를 설정합니다.</span>';
        echo '</div>';
        
        echo '<div class="img-option-row">';
        echo '<label for="pixabay-per-page">페이지당 결과 수:</label> ';
        echo '<input type="number" id="pixabay-per-page" name="img_manage_options[pixabay_per_page]" value="' . esc_attr($per_page) . '" class="img-number" min="3" max="200">';
        echo '<span class="img-desc">검색 시 한 페이지에 표시할 이미지 수입니다. (3~200)</span>';
        echo '</div>';
    }
    
    /**
     * 관리자 설정 페이지
     */
    public function options_page() {
        // 관리자 권한 확인
        if (!current_user_can('manage_options')) {
            wp_die('이 페이지에 접근할 권한이 없습니다.');
        }
        
        // 설정 저장 메시지 표시
        if (isset($_GET['settings-updated']) && $_GET['settings-updated']) {
            add_settings_error('img_manage_messages', 'img_manage_message', '설정이 저장되었습니다.', 'updated');
        }
        
        // 현재 설정 가져오기
        $options = get_option('img_manage_options', img_manage_default_options());
        ?>
        <div class="wrap img-manage-admin">
            <h1><span class="dashicons dashicons-format-image"></span> 이미지 관리</h1>
            
            <?php settings_errors('img_manage_messages'); ?>
            
            <div class="nav-tab-wrapper img-tabs">
                <a href="#settings" class="nav-tab nav-tab-active">설정</a>
                <a href="#pixabay" class="nav-tab">Pixabay</a>
                <a href="#web-vitals" class="nav-tab">Core Web Vitals</a>
                <a href="#help" class="nav-tab">도움말</a>
            </div>
            
            <form method="post" action="options.php">
                <?php settings_fields('img_manage_options_group'); ?>
                
                <div id="settings" class="tab-content" style="display: block;">
                    <div class="img-admin-grid">
                        <!-- 이미지 기본 설정 카드 -->
                        <div class="img-admin-card">
                            <h2 class="img-card-title">이미지 기본 설정</h2>
                            <div class="img-card-content">
                                <?php do_settings_fields('image-management', 'img_manage_main_section'); ?>
                            </div>
                        </div>
                        
                        <!-- 이미지 변환 설정 카드 - 우측에 배치 -->
                        <div class="img-admin-card">
                            <h2 class="img-card-title">이미지 변환 설정</h2>
                            <div class="img-card-content">
                                <?php do_settings_fields('image-management', 'img_manage_conversion_section'); ?>
                            </div>
                        </div>
                        
                        <!-- SEO 최적화 설정 카드 -->
                        <div class="img-admin-card">
                            <h2 class="img-card-title">SEO 최적화 설정</h2>
                            <div class="img-card-content">
                                <?php do_settings_fields('image-management', 'img_manage_seo_section'); ?>
                            </div>
                        </div>
                        
                        <!-- 반응형 이미지 설정 카드 -->
                        <div class="img-admin-card">
                            <h2 class="img-card-title">반응형 이미지 설정</h2>
                            <div class="img-card-content">
                                <?php do_settings_fields('image-management', 'img_manage_responsive_section'); ?>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div id="pixabay" class="tab-content" style="display: none;">
                    <div class="img-admin-card">
                        <h2 class="img-card-title">Pixabay API 설정</h2>
                        <div class="img-card-content">
                            <?php do_settings_fields('image-management', 'img_manage_pixabay_section'); ?>
                            
                            <div class="pixabay-info">
                                <h3>Pixabay 통합 사용법</h3>
                                <p>이 기능을 활성화하면 글 작성 화면에서 Pixabay의 무료 이미지를 검색하고 바로 삽입할 수 있습니다.</p>
                                
                                <div class="pixabay-usage">
                                    <h4>사용 방법:</h4>
                                    <ol>
                                        <li>글 편집 화면에서 <strong>Pixabay 이미지 검색</strong> 버튼을 클릭합니다.</li>
                                        <li>원하는 검색어를 입력하고 필터를 설정합니다.</li>
                                        <li>검색 결과에서 원하는 이미지를 클릭하면 자동으로 다운로드되어 글에 삽입됩니다.</li>
                                        <li>다운로드된 이미지는 자동으로 WebP 형식으로 변환됩니다.</li>
                                    </ol>
                                </div>
                                
                                <div class="pixabay-attribution">
                                    <p><strong>중요:</strong> Pixabay에서 제공하는 이미지는 무료로 사용할 수 있지만, 가능한 경우 출처를 명시하는 것이 좋습니다.</p>
                                    <p>이미지 제공: <a href="https://pixabay.com/" target="_blank">Pixabay</a></p>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div id="web-vitals" class="tab-content" style="display: none;">
                    <div class="img-admin-card">
                        <h2 class="img-card-title">Core Web Vitals 최적화 설정</h2>
                        <div class="img-card-content">
                            <?php do_settings_fields('image-management', 'img_manage_web_vitals_section'); ?>
                            
                            <div class="web-vitals-info">
                                <h3>Core Web Vitals란?</h3>
                                <p>Core Web Vitals는 Google이 웹사이트의 사용자 경험을 측정하기 위해 사용하는 핵심 지표입니다. 이 설정들은 이미지 관련 성능을 최적화하여 점수를 향상시킵니다.</p>
                                
                                <div class="web-vitals-metrics">
                                    <h4>주요 지표:</h4>
                                    <ul>
                                        <li><strong>LCP (Largest Contentful Paint)</strong>: 페이지 로딩 속도를 측정합니다. 이미지 최적화와 직접적인 관련이 있습니다.</li>
                                        <li><strong>CLS (Cumulative Layout Shift)</strong>: 시각적 안정성을 측정합니다. 이미지에 치수를 명시하면 개선됩니다.</li>
                                        <li><strong>FID (First Input Delay)</strong>: 상호작용 지연 시간을 측정합니다.</li>
                                    </ul>
                                </div>
                                
                                <div class="optimization-tips">
                                    <h4>최적화 팁:</h4>
                                    <ul>
                                        <li>모든 이미지에 <code>width</code>와 <code>height</code> 속성을 명시하세요.</li>
                                        <li>중요한 이미지(화면 상단)는 지연 로딩에서 제외하세요.</li>
                                        <li>적절한 이미지 형식과 압축률을 사용하세요.</li>
                                        <li>반응형 이미지 기술(srcset, sizes)을 활용하세요.</li>
                                    </ul>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div id="help" class="tab-content" style="display: none;">
                    <div class="img-admin-grid">
                        <!-- 통합된 도움말 카드 (좌측) -->
                        <div class="img-admin-card">
                            <h2 class="img-card-title">이미지 최적화 안내</h2>
                            <div class="img-card-content">
                                <p>이미지 최적화는 웹사이트의 성능을 향상시키고 로딩 시간을 단축하기 위해 이미지 파일의 크기를 줄이는 과정입니다.</p>
                                
                                <h3>SEO와 이미지 최적화</h3>
                                <p>이 플러그인은 다음과 같은 SEO 최적화 기능을 제공합니다:</p>
                                <ul class="img-bullet-list">
                                    <li><strong>SEO 친화적인 파일명</strong>: 원본 파일명을 최대한 유지하면서 고유성을 보장합니다.</li>
                                    <li><strong>최적화된 alt 태그</strong>: 본문 이미지와 썸네일에 대해 최적화된 alt 태그 제공</li>
                                    <li><strong>메타데이터 보존</strong>: 이미지의 EXIF, IPTC 등 중요한 메타데이터를 유지합니다.</li>
                                    <li><strong>WebP 형식 지원</strong>: 더 작은 파일 크기로 높은 품질을 제공하는 최신 이미지 형식을 사용합니다.</li>
                                </ul>
                                
                                <h3>권장 설정</h3>
                                <ul class="img-bullet-list">
                                    <li><strong>이미지 형식 변환:</strong>
                                        <ul>
                                            <li>PNG → WebP (80-85% 품질): 투명도를 유지하면서 파일 크기 감소</li>
                                            <li>JPG → WebP (85% 품질): 시각적 품질 유지하면서 파일 크기 감소</li>
                                        </ul>
                                    </li>
                                    <li><strong>이미지 크기:</strong>
                                        <ul>
                                            <li>최대 가로크기: 1920px (대형 디스플레이)</li>
                                            <li>중간 가로크기: 1200px (태블릿)</li>
                                            <li>작은 가로크기: 768px (모바일)</li>
                                            <li>썸네일 가로크기: 400px (미리보기)</li>
                                        </ul>
                                    </li>
                                </ul>
                                
                                <h3>Pixabay 통합</h3>
                                <p>최신 버전에서는 Pixabay API를 통한 무료 이미지 검색 및 추가 기능을 제공합니다:</p>
                                <ul class="img-bullet-list">
                                    <li>글 작성 중 쉽게 무료 이미지 검색 및 삽입</li>
                                    <li>다운로드된 이미지를 자동으로 WebP로 변환</li>
                                    <li>이미지 SEO 최적화 자동 적용</li>
                                </ul>
                            </div>
                        </div>
                        
                        <!-- 이미지 크기 처리 예시 카드 (우측) -->
                        <div class="img-admin-card">
                            <h2 class="img-card-title">이미지 크기 처리 예시</h2>
                            <div class="img-card-content">
                                <p>"이미지 크기가 지정한 최대 가로크기보다 작으면 리사이즈하지 않고 형식만 변환합니다"라는 설정의 의미를 예시로 살펴보겠습니다.</p>
                                
                                <h4>기본 설정값:</h4>
                                <ul class="img-bullet-list">
                                    <li>최대 가로크기: 1920픽셀</li>
                                    <li>중간 가로크기: 1200픽셀</li>
                                    <li>작은 가로크기: 768픽셀</li>
                                    <li>썸네일 가로크기: 400픽셀</li>
                                </ul>
                                
                                <h4>예시 1: 큰 이미지 (2500픽셀 너비)</h4>
                                <ul class="img-bullet-list">
                                    <li>원본 이미지: 2500px × 1500px PNG 파일</li>
                                    <li>처리 결과:
                                        <ol style="margin-left: 20px;">
                                            <li>이미지가 최대 가로크기(1920픽셀)보다 크므로 리사이즈됨</li>
                                            <li>1920px × 1152px 크기로 리사이즈됨 (비율 유지)</li>
                                            <li>PNG에서 WebP로 형식 변환됨</li>
                                            <li>추가로 1200px, 768px, 400px 크기의 버전도 생성됨 (반응형 이미지용)</li>
                                        </ol>
                                    </li>
                                </ul>
                                
                                <h4>예시 2: 중간 크기 이미지 (1500픽셀 너비)</h4>
                                <ul class="img-bullet-list">
                                    <li>원본 이미지: 1500px × 900px JPG 파일</li>
                                    <li>처리 결과:
                                        <ol style="margin-left: 20px;">
                                            <li>이미지가 최대 가로크기(1920픽셀)보다 작으므로 리사이즈되지 않음</li>
                                            <li>원본 크기(1500px × 900px) 유지</li>
                                            <li>JPG에서 WebP로 형식만 변환됨</li>
                                            <li>추가로 1200px, 768px, 400px 크기의 버전도 생성됨 (반응형 이미지용)</li>
                                        </ol>
                                    </li>
                                </ul>
                                
                                <h3>새로운 기능: 픽사베이 통합</h3>
                                <p>버전 1.1부터 추가된 Pixabay 통합 기능은 다음과 같은 이점을 제공합니다:</p>
                                <ul class="img-bullet-list">
                                    <li>고품질 무료 이미지에 쉽게 접근</li>
                                    <li>글 작성 흐름을 방해하지 않는 직관적인 인터페이스</li>
                                    <li>자동 이미지 최적화 적용 (WebP 변환, 리사이즈, alt 태그 설정)</li>
                                    <li>저작권 문제 없는 이미지 사용</li>
                                </ul>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div class="submit-button-container">
                    <?php submit_button('설정 저장', 'primary', 'submit', false, ['class' => 'img-submit-btn']); ?>
                </div>
            </form>
            
            <!-- 탭 전환 JavaScript -->
            <script>
            jQuery(document).ready(function($) {
                // 탭 기능
                $('.nav-tab').on('click', function(e) {
                    e.preventDefault();
                    
                    // 탭 활성화
                    $('.nav-tab').removeClass('nav-tab-active');
                    $(this).addClass('nav-tab-active');
                    
                    // 콘텐츠 표시
                    $('.tab-content').hide();
                    $($(this).attr('href')).show();
                });
            });
            </script>
        </div>
        <?php
    }
}