<?php
/*
 * 파일명: image-format-converter.php
 * 위치: /wp-content/plugins/image-format-converter-plus/
 * 기능: 업로드된 PNG, JPG, WEBP 이미지를 자동 변환/압축 및 반응형 리사이즈, Pixabay 무료 이미지 통합
 * 작성일: 2025-04-19
 * 수정일: 2025-12-30
 */

/*
  Plugin Name: Image Format Converter Plus
  Plugin URI: https://url.kr/p/wp-plugin/image-format-converter-plus
  Description: PNG, JPG 파일을 WebP 형식으로 변환하고 이미지 크기를 자동 조정하며, Core Web Vitals 점수 향상을 위한 다양한 기능을 제공합니다. 또한 Pixabay API를 통해 무료 이미지를 쉽게 검색하고 추가할 수 있는 기능을 제공합니다.
  Version: 2.0
  Author: 호텔천사
  Author URI: https://webworks.co.kr
  Text Domain: image-format-converter-plus
 */

// 직접 접근 방지
if (!defined('ABSPATH')) {
    exit;
}

// ===================================
// 1. 초기화 및 상수 정의
// ===================================

// 플러그인 경로 상수 정의
define('IMG_MANAGE_PATH', plugin_dir_path(__FILE__));
define('IMG_MANAGE_URL', plugin_dir_url(__FILE__));
define('IMG_MANAGE_VERSION', '2.0');

// 기본 설정값 정의
function img_manage_default_options() {
    return array(
        'max_image_width' => 1920,
        'medium_image_width' => 1200,
        'small_image_width' => 768,
        'thumbnail_width' => 400,
        'skip_small_images' => 1,
        'preserve_metadata' => 1,
        'enable_srcset' => 1,
        'png_enabled' => 1,
        'png_target' => 'webp',
        'png_quality' => 80,
        'jpg_enabled' => 1,
        'jpg_quality' => 85,
        'webp_enabled' => 0,
        'webp_quality' => 85,
        'seo_filename' => 1,
        'lazy_loading' => 1,
        'add_dimensions' => 1,
        'prioritize_lcp' => 1,
        'pixabay_enabled' => 1,
		'pixabay_api_key' => '33793945-', // 기본 API 키 설정
        'pixabay_default_type' => 'photo',
        'pixabay_default_orientation' => 'all',
        'pixabay_default_category' => '',
        'pixabay_per_page' => 20,
    );
}

// 플러그인 활성화 시 기본 옵션 설정
function img_manage_activate() {
    if (!get_option('img_manage_options')) {
        add_option('img_manage_options', img_manage_default_options());
    }

    // 이미지 저장을 위한 디렉토리 생성
    $upload_dir = wp_upload_dir();
    $pixabay_dir = $upload_dir['basedir'] . '/pixabay-images';

    if (!file_exists($pixabay_dir)) {
        wp_mkdir_p($pixabay_dir);

        // 디렉토리 보호를 위한 index.php 파일 생성
        $index_file = $pixabay_dir . '/index.php';
        if (!file_exists($index_file)) {
            file_put_contents($index_file, '<?php // Silence is golden');
        }
    }
}

register_activation_hook(__FILE__, 'img_manage_activate');

// 플러그인 비활성화 시 정리 작업
function img_manage_deactivate() {
    // 정리 작업이 필요한 경우 여기에 추가
}
register_deactivation_hook(__FILE__, 'img_manage_deactivate');

// 플러그인 제거 시 정리 작업
function img_manage_uninstall() {
    // 옵션 삭제
    delete_option('img_manage_options');

    // 업로드된 Pixabay 이미지를 삭제할지 여부는 선택사항
    // 여기서는 삭제하지 않음 (사용자 콘텐츠 보존)
}
register_uninstall_hook(__FILE__, 'img_manage_uninstall');

// ===================================
// 2. 필요한 파일 포함
// ===================================

// 이미지 변환 클래스
require_once(IMG_MANAGE_PATH . 'includes/class-image-converter.php');

// Pixabay API 통합 클래스
require_once(IMG_MANAGE_PATH . 'includes/class-pixabay-api.php');

// 관리자 설정 페이지 클래스
require_once(IMG_MANAGE_PATH . 'includes/class-admin-settings.php');

// Core Web Vitals 최적화 클래스
require_once(IMG_MANAGE_PATH . 'includes/class-core-web-vitals.php');

// ===================================
// 3. 필요한 스크립트 및 스타일 로드
// ===================================

// 관리자 스크립트 및 스타일 로드
function img_manage_admin_scripts($hook) {
    // 플러그인 설정 페이지나 포스트 편집 페이지에서만 로드
    if ($hook == 'toplevel_page_image-management' || $hook == 'post.php' || $hook == 'post-new.php') {
        // 관리자 스타일 로드
        wp_enqueue_style(
            'img-manage-admin-style',
            IMG_MANAGE_URL . 'assets/css/admin-style.css',
            array(),
            IMG_MANAGE_VERSION
        );

        // 관리자 스크립트 로드
        wp_enqueue_script(
            'img-manage-admin-script',
            IMG_MANAGE_URL . 'assets/js/admin-script.js',
            array('jquery'),
            IMG_MANAGE_VERSION,
            true
        );
    }

    // 포스트 편집 페이지에서만 Pixabay 관련 스크립트 로드
    if ($hook == 'post.php' || $hook == 'post-new.php') {
        $options = get_option('img_manage_options');

        // Pixabay 기능이 활성화된 경우에만 로드
        if (isset($options['pixabay_enabled']) && $options['pixabay_enabled'] == 1 && !empty($options['pixabay_api_key'])) {
            // Pixabay 모달 스타일
            wp_enqueue_style(
                'img-manage-pixabay-modal',
                IMG_MANAGE_URL . 'assets/css/pixabay-modal.css',
                array(),
                IMG_MANAGE_VERSION
            );

            // Pixabay 검색 스크립트
            wp_enqueue_script(
                'img-manage-pixabay-search',
                IMG_MANAGE_URL . 'assets/js/pixabay-search.js',
                array('jquery'),
                IMG_MANAGE_VERSION,
                true
            );

            // Gutenberg 통합 스크립트
            if (function_exists('use_block_editor_for_post_type') && use_block_editor_for_post_type(get_post_type())) {
                wp_enqueue_script(
                    'img-manage-gutenberg',
                    IMG_MANAGE_URL . 'assets/js/gutenberg-integration.js',
                    array('wp-blocks', 'wp-element', 'wp-editor', 'wp-components', 'wp-i18n', 'wp-compose', 'wp-data', 'wp-plugins', 'wp-edit-post', 'jquery'),
                    IMG_MANAGE_VERSION,
                    true
                );

                // Pixabay API 설정을 스크립트로 전달
                wp_localize_script('img-manage-gutenberg', 'imgManagePixabay', array(
                    'apiKey' => $options['pixabay_api_key'],
                    'ajaxUrl' => admin_url('admin-ajax.php'),
                    'nonce' => wp_create_nonce('img-manage-pixabay'),
                    'settings' => array(
                        'defaultType' => $options['pixabay_default_type'] ?? 'photo',
                        'defaultOrientation' => $options['pixabay_default_orientation'] ?? 'all',
                        'defaultCategory' => $options['pixabay_default_category'] ?? '',
                        'perPage' => $options['pixabay_per_page'] ?? 20
                    )
                ));

                // 픽사베이 설정을 검색 스크립트에도 전달
                wp_localize_script('img-manage-pixabay-search', 'imgManagePixabay', array(
                    'apiKey' => $options['pixabay_api_key'],
                    'ajaxUrl' => admin_url('admin-ajax.php'),
                    'nonce' => wp_create_nonce('img-manage-pixabay'),
                    'settings' => array(
                        'defaultType' => $options['pixabay_default_type'] ?? 'photo',
                        'defaultOrientation' => $options['pixabay_default_orientation'] ?? 'all',
                        'defaultCategory' => $options['pixabay_default_category'] ?? '',
                        'perPage' => $options['pixabay_per_page'] ?? 20
                    )
                ));
            }
        }
    }
}

add_action('admin_enqueue_scripts', 'img_manage_admin_scripts');

// 필요한 클래스 초기화
function img_manage_init() {
    // 이미지 변환 클래스 초기화
    $image_converter = new Image_Converter();

    // Pixabay API 통합 클래스 초기화
    $pixabay_api = new Pixabay_API();

    // 관리자 설정 페이지 클래스 초기화
    $admin_settings = new Admin_Settings();

    // Core Web Vitals 최적화 클래스 초기화
    $core_web_vitals = new Core_Web_Vitals();
}
add_action('init', 'img_manage_init');