/**
 * 파일명: pixabay-search.js
 * 위치: /wp-content/plugins/image-format-converter/assets/js/
 * 기능: Pixabay 이미지 검색 모달 스크립트
 * 작성일: 2025-04-21
 * 수정일: 2025-04-21
 */

(function($) {
    'use strict';
    
    // 전역 변수
    let currentPage = 1;
    let totalHits = 0;
    let currentQuery = '';
    let currentImageType = 'photo';
    let currentOrientation = 'all';
    let currentCategory = '';
    let isLoading = false;
    
    // DOM 로드 완료 시
    $(document).ready(function() {
        initPixabayModal();
    });
    
    // Pixabay 모달 초기화
    function initPixabayModal() {
        // 모달 요소
        const $modal = $('#pixabay-modal');
        if (!$modal.length) {
            console.error('Pixabay 모달을 찾을 수 없습니다.');
            
            // 모달이 없으면 생성
            createPixabayModal();
            return;
        }
        
        const $closeBtn = $('.pixabay-modal-close');
        
        // 검색 요소
        const $searchInput = $('#pixabay-search-input');
        const $searchButton = $('#pixabay-search-button');
        const $imageTypeSelect = $('#pixabay-image-type');
        const $orientationSelect = $('#pixabay-orientation');
        const $categorySelect = $('#pixabay-category');
        
        // 결과 요소
        const $imagesContainer = $('#pixabay-images-container');
        const $resultsCount = $('#pixabay-results-count');
        const $loadMoreButton = $('#pixabay-load-more-button');
        const $loadingIndicator = $('#pixabay-loading');
        const $noResults = $('#pixabay-no-results');
        const $searchResults = $('.pixabay-search-results');
        const $loadMore = $('.pixabay-load-more');
        
        // 기본값 설정 (imgManagePixabay 객체가 있는 경우)
        if (typeof imgManagePixabay !== 'undefined') {
            if ($imageTypeSelect.length && imgManagePixabay.settings && imgManagePixabay.settings.defaultType) {
                $imageTypeSelect.val(imgManagePixabay.settings.defaultType);
                currentImageType = imgManagePixabay.settings.defaultType;
            }
            
            if ($orientationSelect.length && imgManagePixabay.settings && imgManagePixabay.settings.defaultOrientation) {
                $orientationSelect.val(imgManagePixabay.settings.defaultOrientation);
                currentOrientation = imgManagePixabay.settings.defaultOrientation;
            }
            
            if ($categorySelect.length && imgManagePixabay.settings && imgManagePixabay.settings.defaultCategory) {
                $categorySelect.val(imgManagePixabay.settings.defaultCategory);
                currentCategory = imgManagePixabay.settings.defaultCategory;
            }
        }
        
        // 모달 열기 (커스텀 이벤트 리스너)
        $(document).on('open-pixabay-modal', function() {
            console.log('Pixabay 모달 열기 이벤트 발생');
            $modal.show();
            $searchInput.focus();
            
            // 필터 값 초기화 (select 요소가 있는 경우)
            if ($imageTypeSelect.length) currentImageType = $imageTypeSelect.val();
            if ($orientationSelect.length) currentOrientation = $orientationSelect.val();
            if ($categorySelect.length) currentCategory = $categorySelect.val();
        });
        
        // 모달 닫기
        $closeBtn.on('click', function() {
            $modal.hide();
            resetSearchResults();
        });
        
        // 모달 외부 클릭 시 닫기
        $(window).on('click', function(event) {
            if (event.target === $modal[0]) {
                $modal.hide();
                resetSearchResults();
            }
        });
        
        // 검색 실행
        $searchButton.on('click', function() {
            executeSearch();
        });
        
        // 엔터 키로 검색
        $searchInput.on('keypress', function(e) {
            if (e.which === 13) {
                executeSearch();
            }
        });
        
        // 필터 변경 시 검색 자동 실행 (이미 검색한 경우에만)
        if ($imageTypeSelect.length) {
            $imageTypeSelect.on('change', function() {
                currentImageType = $(this).val();
                if (currentQuery) {
                    resetSearchResults();
                    executeSearch();
                }
            });
        }
        
        if ($orientationSelect.length) {
            $orientationSelect.on('change', function() {
                currentOrientation = $(this).val();
                if (currentQuery) {
                    resetSearchResults();
                    executeSearch();
                }
            });
        }
        
        if ($categorySelect.length) {
            $categorySelect.on('change', function() {
                currentCategory = $(this).val();
                if (currentQuery) {
                    resetSearchResults();
                    executeSearch();
                }
            });
        }
        
        // 더 불러오기 버튼
        $loadMoreButton.on('click', function() {
            loadMoreImages();
        });
        
        /**
         * 검색 실행 함수
         */
        function executeSearch() {
            const query = $searchInput.val().trim();
            
            if (!query) {
                return;
            }
            
            // 검색 결과 초기화
            resetSearchResults();
            
            // 검색 쿼리와 필터 값 저장
            currentQuery = query;
            if ($imageTypeSelect.length) currentImageType = $imageTypeSelect.val();
            if ($orientationSelect.length) currentOrientation = $orientationSelect.val();
            if ($categorySelect.length) currentCategory = $categorySelect.val();
            currentPage = 1;
            
            // 이미지 검색 실행
            searchPixabayImages();
        }
        
        /**
         * Pixabay API 검색 함수
         */
        function searchPixabayImages() {
            if (isLoading) {
                return;
            }
            
            // 로딩 상태 표시
            isLoading = true;
            $loadingIndicator.show();
            $noResults.hide();
            $loadMore.hide();
            
            // API URL 구성
            let apiUrl;
            let requestData;
            
            if (typeof imgManagePixabay !== 'undefined' && imgManagePixabay.ajaxUrl) {
                // WordPress AJAX 사용
                apiUrl = imgManagePixabay.ajaxUrl;
                requestData = {
                    action: 'img_manage_pixabay_search',
                    nonce: imgManagePixabay.nonce,
                    query: currentQuery,
                    image_type: currentImageType,
                    orientation: currentOrientation,
                    category: currentCategory,
                    per_page: 20,
                    page: currentPage
                };
            } else {
                // Pixabay API 직접 호출 (프로젝트 폴더가 제대로 설정되지 않은 경우)
                console.error('imgManagePixabay 객체가 정의되지 않았습니다. 직접 API 호출을 시도합니다.');
                
                // 직접 API 호출 시 발생할 수 있는 CORS 문제 경고
                console.warn('직접 API 호출은 CORS 정책으로 인해 실패할 수 있습니다.');
                
                apiUrl = 'https://pixabay.com/api/';
                requestData = {
                    key: '33793945-73a3c6fb2ea7f30d436f3acea', // 하드코딩된 API 키 (비상용)
                    q: currentQuery,
                    image_type: currentImageType,
                    orientation: currentOrientation,
                    category: currentCategory,
                    per_page: 20,
                    page: currentPage,
                    safesearch: true
                };
            }
            
            // AJAX 요청
            $.ajax({
                url: apiUrl,
                type: 'GET',
                data: requestData,
                success: function(response) {
                    isLoading = false;
                    $loadingIndicator.hide();
                    
                    let data;
                    
                    // WordPress AJAX 응답 처리
                    if (response.success !== undefined) {
                        data = response.data;
                    } else {
                        // 직접 API 호출 응답 처리
                        data = response;
                    }
                    
                    if (data && data.hits) {
                        const images = data.hits;
                        totalHits = data.totalHits;
                        
                        // 결과가 있는 경우
                        if (images.length > 0) {
                            $searchResults.show();
                            
                            // 결과 개수 표시
                            $resultsCount.text('총 ' + totalHits + '개의 이미지 중 ' + 
                                (currentPage * 20 > totalHits ? 
                                    totalHits : 
                                    currentPage * 20) + '개 표시 중');
                            
                            // 이미지 그리드에 이미지 추가
                            renderImages(images);
                            
                            // 더 불러올 이미지가 있는 경우 더 불러오기 버튼 표시
                            if (currentPage * 20 < totalHits) {
                                $loadMore.show();
                            } else {
                                $loadMore.hide();
                            }
                        } else {
                            // 결과가 없는 경우
                            $noResults.show();
                        }
                    } else {
                        // API 응답 오류
                        $noResults.show();
                        console.error('Pixabay API 응답 오류:', response);
                    }
                },
                error: function(xhr, status, error) {
                    isLoading = false;
                    $loadingIndicator.hide();
                    $noResults.show();
                    console.error('AJAX 요청 오류:', error);
                }
            });
        }
        
        /**
         * 추가 이미지 로드 함수
         */
        function loadMoreImages() {
            currentPage++;
            searchPixabayImages();
        }
        
        /**
         * 이미지 렌더링 함수
         */
        function renderImages(images) {
            images.forEach(function(image) {
                const $imageItem = $('<div>', {
                    class: 'pixabay-image-item',
                    'data-id': image.id,
                    'data-url': image.largeImageURL
                });
                
                const $image = $('<img>', {
                    src: image.previewURL,
                    alt: image.tags,
                    'data-original': image.largeImageURL
                });
                
                const $overlay = $('<div>', {
                    class: 'pixabay-image-overlay',
                    text: image.user + ' | ' + image.imageWidth + 'x' + image.imageHeight
                });
                
                $imageItem.append($image, $overlay);
                $imagesContainer.append($imageItem);
                
                // 이미지 클릭 이벤트
                $imageItem.on('click', function() {
                    downloadAndInsertImage(image);
                });
            });
        }
        
        /**
         * 이미지 다운로드 및 삽입 함수
         */
        function downloadAndInsertImage(image) {
            if (isLoading) {
                return;
            }
            
            // 로딩 상태 표시
            isLoading = true;
            $loadingIndicator.show();
            
            // alt 텍스트 생성
            const altText = 'Pixabay 이미지: ' + image.tags;
            
            // WordPress AJAX 사용 가능 여부 확인
            if (typeof imgManagePixabay === 'undefined' || !imgManagePixabay.ajaxUrl) {
                console.error('imgManagePixabay 객체가 정의되지 않았습니다. 이미지를 다운로드할 수 없습니다.');
                isLoading = false;
                $loadingIndicator.hide();
                alert('WordPress 연결 오류로 이미지를 다운로드할 수 없습니다.');
                return;
            }
            
            // AJAX 요청
            $.ajax({
                url: imgManagePixabay.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'img_manage_pixabay_download',
                    nonce: imgManagePixabay.nonce,
                    image_url: image.largeImageURL,
                    pixabay_id: image.id,
                    alt_text: altText,
                    filename: 'pixabay-' + image.id
                },
                success: function(response) {
                    isLoading = false;
                    $loadingIndicator.hide();
                    
                    if (response.success && response.data && response.data.attachment_id) {
                        // 이미지 삽입
                        insertImageToEditor(response.data.attachment_id, response.data.url, altText);
                        
                        // 모달 닫기
                        $modal.hide();
                        resetSearchResults();
                        
                        // 성공 메시지
                        showNotification('픽사베이 이미지가 미디어 라이브러리에 추가되었습니다.');
                    } else {
                        // 오류 메시지
                        showNotification('이미지 다운로드에 실패했습니다. 다시 시도해주세요.', 'error');
                        console.error('이미지 다운로드 오류:', response);
                    }
                },
                error: function(xhr, status, error) {
                    isLoading = false;
                    $loadingIndicator.hide();
                    showNotification('이미지 다운로드에 실패했습니다. 다시 시도해주세요.', 'error');
                    console.error('AJAX 요청 오류:', error);
                }
            });
        }
        
        /**
         * 에디터에 이미지 삽입 함수
         */
        function insertImageToEditor(attachmentId, imageUrl, altText) {
            // 현재 활성화된 에디터 확인
            if (typeof wp !== 'undefined' && wp.data && wp.data.select('core/editor')) {
                // Gutenberg 에디터
                insertToGutenberg(attachmentId, imageUrl, altText);
            } else {
                // 클래식 에디터
                insertToClassicEditor(attachmentId, imageUrl, altText);
            }
        }
        
        /**
         * Gutenberg 에디터에 이미지 삽입
         */
        function insertToGutenberg(attachmentId, imageUrl, altText) {
            // 미디어 객체 가져오기
            if (typeof wp.media !== 'undefined' && wp.media.attachment) {
                wp.media.attachment(attachmentId).fetch().then(function() {
                    const attachment = wp.media.attachment(attachmentId).toJSON();
                    
                    // 이미지 블록 생성
                    if (wp.blocks && wp.blocks.createBlock) {
                        const imageBlock = wp.blocks.createBlock('core/image', {
                            id: attachmentId,
                            url: attachment.url,
                            alt: altText,
                            caption: '',
                            width: attachment.width,
                            height: attachment.height,
                            sizeSlug: 'large'
                        });
                        
                        // 현재 선택된 블록 위치에 삽입
                        if (wp.data && wp.data.dispatch && wp.data.select) {
                            const { insertBlock } = wp.data.dispatch('core/block-editor');
                            const { getBlockInsertionPoint } = wp.data.select('core/block-editor');
                            
                            if (insertBlock && getBlockInsertionPoint) {
                                const insertionPoint = getBlockInsertionPoint();
                                
                                insertBlock(
                                    imageBlock,
                                    insertionPoint.index,
                                    insertionPoint.rootClientId
                                );
                            } else {
                                console.error('블록 삽입 함수를 찾을 수 없습니다.');
                                fallbackInsert(imageUrl, altText);
                            }
                        } else {
                            console.error('wp.data 객체를 찾을 수 없습니다.');
                            fallbackInsert(imageUrl, altText);
                        }
                    } else {
                        console.error('wp.blocks 객체를 찾을 수 없습니다.');
                        fallbackInsert(imageUrl, altText);
                    }
                });
            } else {
                console.error('wp.media 객체를 찾을 수 없습니다.');
                fallbackInsert(imageUrl, altText);
            }
        }
        
        /**
         * 클래식 에디터에 이미지 삽입
         */
        function insertToClassicEditor(attachmentId, imageUrl, altText) {
            // 에디터 인스턴스 가져오기
            if (typeof tinyMCE !== 'undefined' && tinyMCE.get('content')) {
                const editor = tinyMCE.get('content');
                
                if (editor && !editor.isHidden()) {
                    // 비주얼 에디터
                    const html = '<img src="' + imageUrl + '" alt="' + altText + '" class="size-large" />';
                    editor.execCommand('mceInsertContent', false, html);
                } else {
                    // 텍스트 에디터
                    const html = '<img src="' + imageUrl + '" alt="' + altText + '" class="size-large" />';
                    const $content = $('#content');
                    
                    if ($content.length) {
                        const caretPos = $content[0].selectionStart;
                        const currentContent = $content.val();
                        const newContent = currentContent.substring(0, caretPos) + html + currentContent.substring(caretPos);
                        $content.val(newContent);
                    } else {
                        console.error('콘텐츠 영역을 찾을 수 없습니다.');
                        fallbackInsert(imageUrl, altText);
                    }
                }
            } else {
                console.error('tinyMCE 객체를 찾을 수 없습니다.');
                fallbackInsert(imageUrl, altText);
            }
        }
        
        /**
         * 이미지 삽입 대체 기능 (에디터 통합 실패 시)
         */
        function fallbackInsert(imageUrl, altText) {
            // 삽입 실패 시 알림 표시
            alert('에디터에 이미지를 삽입할 수 없습니다. 이미지가 미디어 라이브러리에 추가되었으니 직접 삽입해 주세요.');
            
            // 미디어 라이브러리 페이지로 이동 제안
            if (confirm('미디어 라이브러리로 이동하시겠습니까?')) {
                window.open('/wp-admin/upload.php', '_blank');
            }
        }
        
        /**
         * 알림 표시 함수
         */
        function showNotification(message, type = 'success') {
            if (typeof wp !== 'undefined' && wp.data && wp.data.dispatch('core/notices')) {
                // Gutenberg 알림
                const method = type === 'success' ? 'createSuccessNotice' : 'createErrorNotice';
                wp.data.dispatch('core/notices')[method](message, {
                    type: 'snackbar',
                    isDismissible: true
                });
            } else {
                // 클래식 에디터 알림 (워드프레스 기본 알림 사용)
                const $messageDiv = $('<div>', {
                    class: 'notice notice-' + type + ' is-dismissible',
                    css: {
                        position: 'fixed',
                        top: '40px',
                        right: '20px',
                        zIndex: 99999,
                        padding: '10px 15px'
                    }
                });
                
                const $message = $('<p>', {
                    text: message
                });
                
                const $closeButton = $('<button>', {
                    type: 'button',
                    class: 'notice-dismiss',
                    html: '<span class="screen-reader-text">이 알림 무시하기</span>'
                });
                
                $closeButton.on('click', function() {
                    $messageDiv.fadeOut(300, function() {
                        $(this).remove();
                    });
                });
                
                $messageDiv.append($message, $closeButton);
                $('body').append($messageDiv);
                
                // 3초 후 자동 제거
                setTimeout(function() {
                    $messageDiv.fadeOut(300, function() {
                        $(this).remove();
                    });
                }, 3000);
            }
        }
        
        /**
         * 검색 결과 초기화 함수
         */
        function resetSearchResults() {
            $imagesContainer.empty();
            $searchResults.hide();
            $loadMore.hide();
            $noResults.hide();
            $loadingIndicator.hide();
            currentPage = 1;
        }
    }
    
    /**
     * Pixabay 모달 동적 생성 함수
     */
    function createPixabayModal() {
        console.log('Pixabay 모달을 동적으로 생성합니다.');
        
        // 모달 HTML 생성
        const modalHtml = `
        <div id="pixabay-modal" class="pixabay-modal" style="display: none;">
            <div class="pixabay-modal-content">
                <div class="pixabay-modal-header">
                    <span class="pixabay-modal-close">&times;</span>
                    <h2>Pixabay 무료 이미지 검색</h2>
                </div>
                
                <div class="pixabay-modal-body">
                    <div class="pixabay-search-container">
                        <div class="pixabay-search-form">
                            <div class="pixabay-search-input-group">
                                <input type="text" id="pixabay-search-input" placeholder="검색어를 입력하세요...">
                                <button id="pixabay-search-button" class="button button-primary">검색</button>
                            </div>
                            
                            <div class="pixabay-search-filters">
                                <div class="pixabay-filter-group">
                                    <label for="pixabay-image-type">이미지 유형:</label>
                                    <select id="pixabay-image-type">
                                        <option value="photo">사진</option>
                                        <option value="illustration">일러스트레이션</option>
                                        <option value="vector">벡터 그래픽</option>
                                        <option value="all">모든 이미지</option>
                                    </select>
                                </div>
                                
                                <div class="pixabay-filter-group">
                                    <label for="pixabay-orientation">방향:</label>
                                    <select id="pixabay-orientation">
                                        <option value="all">모든 방향</option>
                                        <option value="horizontal">가로 방향</option>
                                        <option value="vertical">세로 방향</option>
                                    </select>
                                </div>
                                
                                <div class="pixabay-filter-group">
                                    <label for="pixabay-category">카테고리:</label>
                                    <select id="pixabay-category">
                                        <option value="">카테고리 선택 없음</option>
                                        <option value="backgrounds">배경</option>
                                        <option value="fashion">패션</option>
                                        <option value="nature">자연</option>
                                        <option value="science">과학</option>
                                        <option value="education">교육</option>
                                        <option value="health">건강</option>
                                        <option value="people">사람</option>
                                        <option value="religion">종교</option>
                                        <option value="places">장소</option>
                                        <option value="animals">동물</option>
                                        <option value="industry">산업</option>
                                        <option value="computer">컴퓨터</option>
                                        <option value="food">음식</option>
                                        <option value="sports">스포츠</option>
                                        <option value="transportation">교통</option>
                                        <option value="travel">여행</option>
                                        <option value="buildings">건물</option>
                                        <option value="business">비즈니스</option>
                                        <option value="music">음악</option>
                                    </select>
                                </div>
                            </div>
                        </div>
                        
                        <div class="pixabay-search-results" style="display: none;">
                            <div class="pixabay-results-info">
                                <span id="pixabay-results-count"></span>
                            </div>
                            
                            <div id="pixabay-images-container" class="pixabay-images-grid"></div>
                            
                            <div class="pixabay-load-more" style="display: none;">
                                <button id="pixabay-load-more-button" class="button button-secondary">더 많은 이미지 불러오기</button>
                            </div>
                        </div>
                        
                        <div id="pixabay-loading" class="pixabay-loading" style="display: none;">
                            <span class="spinner is-active"></span>
                            <span class="loading-text">이미지 로딩 중...</span>
                        </div>
                        
                        <div id="pixabay-no-results" class="pixabay-no-results" style="display: none;">
                            검색 결과가 없습니다. 다른 검색어로 시도해보세요.
                        </div>
                    </div>
                    
                    <div class="pixabay-attribution">
                        이미지 제공: <a href="https://pixabay.com/" target="_blank" rel="noopener noreferrer">Pixabay</a>
                    </div>
                </div>
            </div>
        </div>
        `;
        
        // 기본 CSS 스타일 추가
        const cssStyles = `
        <style>
            .pixabay-modal {
                display: none;
                position: fixed;
                z-index: 100000;
                left: 0;
                top: 0;
                width: 100%;
                height: 100%;
                background-color: rgba(0, 0, 0, 0.5);
            }
            
            .pixabay-modal-content {
                position: relative;
                background-color: #fff;
                margin: 5% auto;
                padding: 0;
                width: 80%;
                max-width: 1200px;
                max-height: 85vh;
                border-radius: 5px;
                box-shadow: 0 4px 8px rgba(0, 0, 0, 0.2);
                display: flex;
                flex-direction: column;
            }
            
            .pixabay-modal-header {
                padding: 15px 20px;
                border-bottom: 1px solid #e5e5e5;
                display: flex;
                align-items: center;
                justify-content: space-between;
            }
            
            .pixabay-modal-header h2 {
                margin: 0;
                font-size: 20px;
                line-height: 1.4;
                font-weight: 600;
            }
            
            .pixabay-modal-close {
                color: #aaa;
                float: right;
                font-size: 28px;
                font-weight: bold;
                cursor: pointer;
            }
            
            .pixabay-modal-body {
                padding: 20px;
                overflow-y: auto;
                flex-grow: 1;
            }
            
            .pixabay-search-input-group {
                display: flex;
                margin-bottom: 15px;
            }
            
            .pixabay-search-input-group input {
                flex-grow: 1;
                margin-right: 10px;
                padding: 8px 12px;
                font-size: 16px;
            }
            
            .pixabay-search-filters {
                display: flex;
                flex-wrap: wrap;
                gap: 15px;
                margin-bottom: 20px;
            }
            
            .pixabay-filter-group {
                flex-basis: calc(33.33% - 10px);
            }
            
            .pixabay-filter-group label {
                display: block;
                margin-bottom: 5px;
                font-weight: 500;
            }
            
            .pixabay-filter-group select {
                width: 100%;
                padding: 6px 10px;
            }
            
            .pixabay-images-grid {
                display: grid;
                grid-template-columns: repeat(4, 1fr);
                gap: 15px;
                margin-bottom: 20px;
            }
            
            .pixabay-image-item {
                position: relative;
                overflow: hidden;
                border-radius: 4px;
                box-shadow: 0 1px 3px rgba(0, 0, 0, 0.1);
                cursor: pointer;
                transition: transform 0.2s ease-in-out, box-shadow 0.2s ease-in-out;
            }
            
            .pixabay-image-item:hover {
                transform: translateY(-3px);
                box-shadow: 0 4px 6px rgba(0, 0, 0, 0.15);
            }
            
            .pixabay-image-item img {
                width: 100%;
                height: auto;
                display: block;
            }
            
            @media screen and (max-width: 1024px) {
                .pixabay-images-grid {
                    grid-template-columns: repeat(3, 1fr);
                }
            }
            
            @media screen and (max-width: 768px) {
                .pixabay-images-grid {
                    grid-template-columns: repeat(2, 1fr);
                }
                
                .pixabay-filter-group {
                    flex-basis: calc(50% - 7.5px);
                }
            }
        </style>
        `;
        
        // HTML 및 CSS를 body에 추가
        $('body').append(cssStyles + modalHtml);
        
        // 모달이 추가되었으므로 다시 초기화
        setTimeout(function() {
            initPixabayModal();
        }, 100);
    }
    
})(jQuery);