/**
 * 파일명: gutenberg-integration.js
 * 위치: /wp-content/plugins/image-format-converter/assets/js/
 * 기능: Gutenberg 에디터에 Pixabay 이미지 검색 기능 통합
 * 작성일: 2025-04-21
 */

(function(wp) {
    'use strict';
    
    // WordPress 관련 도구들이 로드될 때까지 대기
    document.addEventListener('DOMContentLoaded', function() {
        // WordPress 객체가 있는지 확인
        if (!wp) {
            console.error('WordPress 편집기 API를 찾을 수 없습니다.');
            return;
        }
        
        // 필요한 WordPress 모듈들이 로드되었는지 확인
        if (!wp.plugins || !wp.editPost || !wp.components || !wp.element || !wp.blocks) {
            console.error('필요한 WordPress 모듈이 로드되지 않았습니다.');
            // 모듈이 로드될 때까지 대기
            setTimeout(initPixabayIntegration, 1000);
            return;
        }
        
        // 모든 모듈이 준비되었으면 초기화
        initPixabayIntegration();
    });
    
    function initPixabayIntegration() {
        // 필요한 WordPress 컴포넌트들
        const { registerPlugin } = wp.plugins;
        const { PluginSidebar, PluginSidebarMoreMenuItem } = wp.editPost;
        const { PanelBody, Button, TextControl, SelectControl, Spinner } = wp.components;
        const { useState, useEffect, Fragment } = wp.element;
        const { __ } = wp.i18n || { __: function(text) { return text; } };
        const { dispatch, select } = wp.data;
        
        // 툴바 버튼 추가 (간단한 방법)
        const addToolbarButton = function() {
            const toolbar = document.querySelector('.edit-post-header-toolbar');
            if (!toolbar) return;
            
            // 이미 버튼이 있는지 확인
            if (document.querySelector('.pixabay-toolbar-button')) return;
            
            const buttonContainer = document.createElement('div');
            buttonContainer.className = 'pixabay-toolbar-button';
            buttonContainer.style.marginLeft = '8px';
            
            const button = document.createElement('button');
            button.type = 'button';
            button.className = 'components-button is-primary';
            button.textContent = 'Pixabay 이미지';
            button.addEventListener('click', function() {
                const event = new CustomEvent('open-pixabay-modal');
                document.dispatchEvent(event);
            });
            
            buttonContainer.appendChild(button);
            toolbar.appendChild(buttonContainer);
        };
        
        // 버튼 추가 실행 및 1초마다 확인 (에디터가 완전히 로드된 후)
        addToolbarButton();
        setInterval(addToolbarButton, 1000);
        
        // 사이드바 플러그인 등록
        try {
            registerPlugin('img-manage-pixabay', {
                render: function() {
                    return (
                        React.createElement(
                            Fragment,
                            null,
                            React.createElement(
                                PluginSidebarMoreMenuItem,
                                {
                                    target: 'img-manage-pixabay-sidebar',
                                    icon: 'format-image'
                                },
                                'Pixabay 이미지 검색'
                            ),
                            React.createElement(
                                PluginSidebar,
                                {
                                    name: 'img-manage-pixabay-sidebar',
                                    title: 'Pixabay 이미지 검색',
                                    icon: 'format-image'
                                },
                                React.createElement(
                                    PanelBody,
                                    { title: '이미지 검색', initialOpen: true },
                                    React.createElement(
                                        Button,
                                        {
                                            isPrimary: true,
                                            onClick: function() {
                                                const event = new CustomEvent('open-pixabay-modal');
                                                document.dispatchEvent(event);
                                            }
                                        },
                                        'Pixabay 이미지 검색 열기'
                                    )
                                )
                            )
                        )
                    );
                }
            });
        } catch (error) {
            console.error('Pixabay 플러그인 등록 실패:', error);
        }
    }
})(window.wp);